/*
   Tom'Catalog
   Copyright 2002, Thomas Aufischer

   based on:

   Z'Catalog: a PalmOS database utility
   Copyright 1998, Andrew Plotkin <erkyrath@eblong.com>
   http://www.eblong.com/zarf/zcatalog.html

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <PalmOS.h>
#include "zcatalog.h"
#include "resource.h"

/* The form handler for the Filter form. */

typedef struct FILT_Workfilter_Struct
{
  Int16 type;
  Int16 testis, testlt, teststr;
  Z_DBInfo u;
}
FILT_Workfilter;

static FormPtr filterform = NULL;
static FILT_Workfilter work;

#define mGetObject(id)  \
    (FrmGetObjectPtr(filterform, FrmGetObjectIndex(filterform, (id))))

static void FILT_set_radio_buttons(Int16 type, Boolean redraw);
static void FILT_set_field_by_type(Int16 type, Boolean redraw);

void FILT_filterform_prepare(Int16 type, Z_DBInfo * val)
{
  work.type = type;

  work.testis = TRUE;
  work.teststr = 0;
  work.testlt = TRUE;

  if(val)
  {
    MemMove(&(work.u), val, sizeof(Z_DBInfo));
  }
  else
  {
    MemSet(&(work.u), sizeof(Z_DBInfo), 0);
    work.u.type = '????';
    work.u.creator = '????';
    work.u.size = 1000;
    work.u.records = 10;
    StrCopy(work.u.name, "Name");
    work.u.attributes = 0;
  }

}

static void FILT_filterform_closeout(Z_Filter * filt)
{
  FieldPtr fld;
  MemHandle han = NULL;
  MemPtr ptr = NULL;
  Int16 len = 0;
  Int16 ix;

  filt->active = TRUE;
  filt->type = work.type;

  fld = mGetObject(fld_FilterValue);
  han = FldGetTextHandle(fld);
  if(han)
  {
    ptr = MemHandleLock((MemHandle) han);
    len = FldGetTextLength(fld);
  }

  switch (filt->type)
  {
    case Z_dColumnTypeCreator:
    case Z_dColumnTypeType:
      filt->u.ostype = 0;
      for(ix = 0; ix < 4; ix++)
      {
        char ch;
        if(ix < len)
        {
          /*
             causes compiler error
             ch = ptr[ix];
           */
          ch = ((char *) ptr)[ix];
        }
        else
        {
          ch = ' ';
        }
        filt->u.ostype = (filt->u.ostype << 8) | (ch & 0xff);
      }
      filt->test = work.testis;
      break;

    case Z_dColumnTypeName:
      if(ptr)
      {
        StrNCopy(filt->u.str, ptr, Z_dDBNameLength);
        filt->u.str[Z_dDBNameLength] = '\0';
      }
      else
      {
        filt->u.str[0] = '\0';
      }
      filt->test = work.teststr;
      break;

    case Z_dColumnTypeSize:
    case Z_dColumnTypeRecords:
    case Z_dColumnTypeAttributes:
      if(ptr && len)
      {
        filt->u.num = StrAToI(ptr);
      }
      else
      {
        filt->u.num = 0;
      }
      filt->test = work.testlt;
      break;

    default:
      filt->u.num = 0;
      filt->test = FALSE;
      break;
  }

  if(han)
  {
    MemHandleUnlock((MemHandle) han);
  }

}

static void FILT_filterform_init(FormPtr frm)
{
  char *label;
  ListPtr ls;
  ControlPtr ctl;

  filterform = frm;

  ls = mGetObject(ls_FilterTypePop);
  label = LstGetSelectionText(ls, work.type + 1);
  ctl = mGetObject(pop_FilterTypePop);
  CtlSetLabel(ctl, label);
  LstSetSelection(ls, work.type + 1);

  FILT_set_field_by_type(work.type, FALSE);
  FILT_set_radio_buttons(work.type, FALSE);
}

static void FILT_set_field_by_type(Int16 type, Boolean redraw)
{
  char buf[17];

  switch (type)
  {
    case Z_dColumnTypeCreator:
    case Z_dColumnTypeType:
    {
      UInt32 uval;
      if(type == Z_dColumnTypeCreator)
      {
        uval = work.u.creator;

      }
      else
      {
        uval = work.u.type;
      }
      buf[0] = (uval >> 24) & 0xff;
      buf[1] = (uval >> 16) & 0xff;
      buf[2] = (uval >> 8) & 0xff;
      buf[3] = (uval) & 0xff;
      buf[4] = '\0';
      break;
    }

    case Z_dColumnTypeName:
      StrNCopy(buf, work.u.name, Z_dDBNameLength);
      buf[Z_dDBNameLength] = '\0';
      break;

    case Z_dColumnTypeSize:
      StrIToA(buf, work.u.size);
      break;

    case Z_dColumnTypeRecords:
      StrIToA(buf, work.u.records);
      break;

    case Z_dColumnTypeAttributes:
      StrIToA(buf, work.u.attributes);
      break;

    default:
      buf[0] = '\0';
      break;
  }

  if(StrLen(buf) == 0)
    StrCopy(buf, "value");

  if(StrLen(buf))
  {
    FieldPtr fld;
    MemHandle han, oldhan;
    char *ptr;

    fld = mGetObject(fld_FilterValue);
    oldhan = FldGetTextHandle(fld); /* probably NULL, but anyhow */

    han = (MemHandle) MemHandleNew(StrLen(buf) + 1);
    if(han)
    {
      ptr = MemHandleLock((MemHandle) han);
      if(ptr)
      {
        StrCopy(ptr, buf);
        MemHandleUnlock((MemHandle) han);
        FldSetTextHandle(fld, han);
        if(oldhan)
          MemHandleFree((MemHandle) oldhan);

        if(redraw)
        {
          FldDrawField(fld);
        }
      }
    }
  }

}

static void FILT_set_radio_buttons(Int16 type, Boolean redraw)
{
  FieldPtr fld;
  ControlPtr ctl01, ctl02, ctl11, ctl12, ctl21, ctl22, ctl23;
  Int16 typec;

  switch (type)
  {
    case Z_dColumnTypeCreator:
    case Z_dColumnTypeType:
      typec = 0;
      break;

    case Z_dColumnTypeSize:
    case Z_dColumnTypeRecords:
    case Z_dColumnTypeAttributes:
      typec = 1;
      break;

    case Z_dColumnTypeName:
      typec = 2;
      break;

    default:
      typec = -1;
      break;
  }

  fld = mGetObject(fld_FilterValue);
  if(typec == 0)
  {
    FldSetMaxChars(fld, 4);
  }
  else
  {
    FldSetMaxChars(fld, 16);
  }

  ctl01 = mGetObject(tog_FilterIs);
  ctl02 = mGetObject(tog_FilterIsnt);
  ctl11 = mGetObject(tog_FilterLE);
  ctl12 = mGetObject(tog_FilterGE);
  ctl21 = mGetObject(tog_FilterBegins);
  ctl22 = mGetObject(tog_FilterEnds);
  ctl23 = mGetObject(tog_FilterContains);

  if(!redraw)
  {
    CtlSetValue(ctl01, work.testis == TRUE);
    CtlSetValue(ctl02, work.testis == FALSE);
    CtlSetValue(ctl11, work.testlt == TRUE);
    CtlSetValue(ctl12, work.testlt == FALSE);
    CtlSetValue(ctl21, work.teststr == 0);
    CtlSetValue(ctl22, work.teststr == 1);
    CtlSetValue(ctl23, work.teststr == 2);

    switch (typec)
    {
      case 0:
        CtlSetUsable(ctl01, TRUE);
        CtlSetUsable(ctl02, TRUE);
        break;
      case 1:
        CtlSetUsable(ctl11, TRUE);
        CtlSetUsable(ctl12, TRUE);
        break;
      case 2:
        CtlSetUsable(ctl21, TRUE);
        CtlSetUsable(ctl22, TRUE);
        CtlSetUsable(ctl23, TRUE);
        break;
    }
    return;
  }

  CtlHideControl(ctl01);
  CtlHideControl(ctl02);
  CtlHideControl(ctl11);
  CtlHideControl(ctl12);
  CtlHideControl(ctl21);
  CtlHideControl(ctl22);
  CtlHideControl(ctl23);

  switch (typec)
  {
    case 0:
      CtlShowControl(ctl01);
      CtlShowControl(ctl02);
      break;
    case 1:
      CtlShowControl(ctl11);
      CtlShowControl(ctl12);
      break;
    case 2:
      CtlShowControl(ctl21);
      CtlShowControl(ctl22);
      CtlShowControl(ctl23);
      break;
  }
}

Boolean FILT_filterform_hanevent(EventPtr event)
{
  Boolean handled = FALSE;
  ControlPtr ctl;

  switch (event->eType)
  {

    case frmOpenEvent:
      FILT_filterform_init(FrmGetActiveForm());
      FrmDrawForm(filterform);
      /* FrmSetFocus(filterform,
         FrmGetObjectIndex(filterform, fld_FilterValue)); */
      handled = TRUE;
      break;

    case frmCloseEvent:
      filterform = NULL;
      /* and pass on */
      break;

    case popSelectEvent:
      if(event->data.popSelect.controlID == pop_FilterTypePop)
      {
        Int16 val = event->data.popSelect.selection - 1;
        if(val != work.type)
        {
          work.type = val;
          FILT_set_field_by_type(work.type, TRUE);
          FILT_set_radio_buttons(work.type, TRUE);
        }
      }
      /* let the standard handler set the selection and label */
      break;

    case ctlSelectEvent:
      switch (event->data.ctlSelect.controlID)
      {

        case btn_FilterCancel:
          FrmReturnToForm(formid_Main);
          handled = TRUE;
          break;

        case btn_FilterDone:
          FILT_filterform_closeout(&SORT_gFilter);  /* copy stuff back */
          FrmReturnToForm(formid_Main);
          MTAB_do_refilter();
          /* Enable unfilter button */
          ctl = FrmGetObjectPtr(MAIN_gMainform,
                                FrmGetObjectIndex(MAIN_gMainform,
                                                  btn_Unfilter));
          CtlShowControl(ctl);
          handled = TRUE;
          break;

        case tog_FilterIs:
          work.testis = TRUE;
          handled = TRUE;
          break;
        case tog_FilterIsnt:
          work.testis = FALSE;
          handled = TRUE;
          break;
        case tog_FilterLE:
          work.testlt = TRUE;
          handled = TRUE;
          break;
        case tog_FilterGE:
          work.testlt = FALSE;
          handled = TRUE;
          break;
        case tog_FilterBegins:
          work.teststr = 0;
          handled = TRUE;
          break;
        case tog_FilterEnds:
          work.teststr = 1;
          handled = TRUE;
          break;
        case tog_FilterContains:
          work.teststr = 2;
          handled = TRUE;
          break;

      }
      break;

    default:
      break;
  }

  return handled;
}
