/*
   Tom'Catalog
   Copyright 2002, Thomas Aufischer

   based on:

   Z'Catalog: a PalmOS database utility
   Copyright 1998, Andrew Plotkin <erkyrath@eblong.com>
   http://www.eblong.com/zarf/zcatalog.html

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <PalmOS.h>
#include <ExgMgr.h>
#include "zcatalog.h"
#include "resource.h"

//in bytes, Hotsynch/PalmOS allows just 4KB !!!
#define dMaxInternalMemoLength (4300)

enum
{
  MAIN_eListFormCardInfo = 1,
  MAIN_eListFormSystemInfo
};

typedef struct appinfo_struct
{
  UInt8 replaceme;
}
appinfo_t;

Boolean MAIN_gBoneheadmode = FALSE;
Boolean MAIN_gUSDateTime = TRUE;
Boolean MAIN_gOS3_avail = FALSE;

FormPtr MAIN_gMainform = NULL;
static Int16 DBI_gDBList_dirty = 0;

// keeps time of last pen-tap
static UInt32 lastTap = 0;

// defines which info should be shown in
static Int16 listFormShow;

static FormPtr listForm = NULL;

/* Static function declarations */

static Int16 initviewstate;
static char initviewname[dmDBNameLength];

static Err init_app(void);
static void final_app(void);
static void app_eventloop(void);
static Boolean app_hanevent(EventPtr event);

static Boolean MAIN_domenu(UInt16 cmd);
static Boolean MAIN_hanevent(EventPtr event);
Boolean MAIN_formListForm_hanevent(EventPtr event);

#define mRebuildAfterDelete                          \
      if( DBI_gDBList_dirty )                        \
      {                                              \
        DBI_load_dbinfo();                           \
        MTAB_maintab_notice_selection( -1, 0 );      \
        DBI_gDBList_dirty = 0;                       \
        MTAB_sanitize_selscroll();                   \
        MTAB_maintab_reload();                       \
        MTAB_maintab_redraw_careful( FALSE );        \
      }

/* Top-level function. */
UInt32 PilotMain(UInt16 cmd, MemPtr cmdPBP, UInt16 launchFlags)
{
  Err err;
  Int32 lx;

  switch (cmd)
  {
    case sysAppLaunchCmdNormalLaunch:
    {
      UInt32 version_needed = sysMakeROMVersion(2, 0, 0, sysROMStageRelease, 0);
      /* That's 0x02003000, to you. */

      err = UTIL_RomVersionCompatible(version_needed, launchFlags);
      if(err)
      {
        return err;
      }

      err = init_app();
      if(err)
      {
        return err;
      }

      if(initviewstate == 1 && (lx = DBI_find_dbi_by_name(initviewname)) >= 0)
      {
        PROP_propform_prepare(lx);
        FrmGotoForm(formid_Properties);
      }
      else
      {
        FrmGotoForm(formid_Main);
      }

      app_eventloop();
      final_app();
      break;
    }

    case sysAppLaunchCmdSyncNotify:
    {
      /* Register the MIME type on syncNotify, so we do not need to
         be run before we can receive data. */
      UInt32 romVersion;
      FtrGet(sysFtrCreator, sysFtrNumROMVersion, &romVersion);
      if(romVersion >= 0x03000000)
      {
        /* MAIN_gOS3_avail... of course we can't check the global. */
        ExgRegisterData(Z_dSelfCreator, exgRegTypeID, Z_dSelfMimetype);
      }
      break;
    }

    case sysAppLaunchCmdExgAskUser:
      /* Default behavior; always ask whether to accept. */
      break;

    case sysAppLaunchCmdExgReceiveData:
    {
      Boolean globavail = ((launchFlags & sysAppLaunchFlagSubCall) != 0);
      err = TOOL_beam_receive_db((ExgSocketPtr) cmdPBP, globavail);
      if(err)
      {
        return err;
      }
      break;
    }
  }

  return 0;
}

static Err init_app()
{
  Z_Preferences prefs;
  UInt16 prefsSize;
  Int16 res;

  if(MAIN_gOS3_avail)
  {
    ExgRegisterData(Z_dSelfCreator, exgRegTypeID, Z_dSelfMimetype);
  }

  /* Initialize the prefs, before loading them. */

#ifdef ZDEBUG
  MAIN_gBoneheadmode = TRUE;
#else
  MAIN_gBoneheadmode = FALSE;
#endif /* ZDEBUG */
  MAIN_gUSDateTime = TRUE;

  initviewstate = 0;
  initviewname[0] = '\0';
  SORT_set_default_column_state();

  /* Read the saved preferences / saved-state information. */

  prefsSize = sizeof(Z_Preferences);
  res =
    PrefGetAppPreferences(Z_dSelfCreator, Z_dPrefsResid, &prefs, &prefsSize,
                          TRUE);

  if(res == noPreferenceFound || res != Z_dPrefsVersion
     || prefsSize < sizeof(Z_Preferences))
  {
    /* not loaded, or not loaded correctly. */
  }
  else
  {
    /* loaded ok */
    MAIN_gBoneheadmode = prefs.bonehead;
    MAIN_gUSDateTime = prefs.usDateTime;

    initviewstate = prefs.viewstate;
    StrCopy(initviewname, prefs.viewname);
    SORT_load_pref_column_state(&prefs);
  }

  DBI_load_dbinfo();
  DBI_gDBList_dirty = 0;

  return 0;
}

static void final_app()
{
  Z_Preferences prefs;

  /* Write the saved preferences / saved-state information.  This data
     will be backed up during a HotSync. */

  prefs.bonehead = MAIN_gBoneheadmode;
  prefs.usDateTime = MAIN_gUSDateTime;

  SORT_unload_pref_column_state(&prefs);
  if(!PROP_propform_unload_state(&prefs))
  {
    prefs.viewstate = 0;
    prefs.viewname[0] = '\0';
  }

  PrefSetAppPreferences(Z_dSelfCreator, Z_dPrefsResid, Z_dPrefsVersion, &prefs,
                        sizeof(Z_Preferences), TRUE);

  DBI_unload_dbinfo();
}

static void app_eventloop()
{
  UInt16 err;
  EventType event;

  do
  {
    EvtGetEvent(&event, evtWaitForever);

    if(SysHandleEvent(&event))
      continue;

    if(MenuHandleEvent(0, &event, &err))
      continue;

    if(app_hanevent(&event))
      continue;

    FrmDispatchEvent(&event);

  }
  while(event.eType != appStopEvent);
}

static Boolean app_hanevent(EventPtr event)
{
  UInt16 formId;
  FormPtr frm;

  if(event->eType == frmLoadEvent)
  {
    /* Load the form resource. */
    formId = event->data.frmLoad.formID;
    frm = FrmInitForm(formId);
    FrmSetActiveForm(frm);

    switch (formId)
    {
      case formid_Main:
        FrmSetEventHandler(frm, MAIN_hanevent);
        break;

      case formid_Filter:
        FrmSetEventHandler(frm, FILT_filterform_hanevent);
        break;

      case formid_Preferences:
        FrmSetEventHandler(frm, PREFS_hanevent);
        break;

      case formid_Properties:
        FrmSetEventHandler(frm, PROP_propform_hanevent);
        break;

      case formid_ListForm:
        FrmSetEventHandler(frm, MAIN_formListForm_hanevent);
        break;
    }

    return TRUE;
  }
  else if(event->eType == appStopEvent )
  {
    DBI_unload_dbinfo();
    FrmCloseAllForms ();
  }

  return FALSE;
}

void MAIN_mainform_dirty_dbist(Int16 val)
{
  /* This is called before GotoForm if the DBI_gDBList needs to be rejiggered. */
  /* 1 for dirty, 2 if we have to deselect. */
  /* ### be smarter, able to say that a single db has changed or vanished? */

  if(val > DBI_gDBList_dirty)
  {
    DBI_gDBList_dirty = val;
  }
}

void MAIN_mainform_notice_new_db(UInt16 card, LocalID lid)
{
  /* This is called at any time if a DB is received via beaming. */

  if(MAIN_gMainform != NULL)
  {
    DBI_load_dbinfo();
    MTAB_maintab_notice_selection(-1, 0);
    DBI_gDBList_dirty = 0;
    MTAB_sanitize_selscroll();
    MTAB_maintab_reload();
    MTAB_maintab_redraw();
  }
  else
  {
    MAIN_mainform_dirty_dbist(2);
  }
}

static void MAIN_gMainform_init(FormPtr frm)
{
  Int16 ix;
  Int16 colmode;
  ControlPtr ctl;

  MAIN_gMainform = frm;

  if(DBI_gDBList_dirty)
  {
    DBI_load_dbinfo();
    if(DBI_gDBList_dirty >= 2)
    {
      MTAB_maintab_notice_selection(-1, 0);
    }
    DBI_gDBList_dirty = 0;
  }

  MTAB_maintab_init();

  MTAB_sanitize_selscroll();
  MTAB_maintab_reload();

  for(ix = 0; ix < Z_dNumColumns; ix++)
  {
    colmode = SORT_gColumnState[ix].type;
    ctl = FrmGetObjectPtr(frm, FrmGetObjectIndex(frm, SORT_gColumnState[ix].pop_resid));
    CtlSetLabel(ctl, Z_cColumnName[colmode]);
  }

  ctl = FrmGetObjectPtr(frm, FrmGetObjectIndex(frm, chk_ShowRO));
  CtlSetValue(ctl, SORT_gShowReadonly);

  /* ctl = FrmGetObjectPtr(frm, FrmGetObjectIndex(frm, btn_Filter)); */

  ctl = FrmGetObjectPtr(frm, FrmGetObjectIndex(frm, btn_Unfilter));
  CtlSetUsable(ctl, SORT_gFilter.active);
}

void MAIN_drawProperties()
{
  Int32 dbi;

  MTAB_maintab_get_selection_dbi(&dbi, NULL);
  if(dbi < 0 || dbi >= DBI_gNumDBs)
  {
    FrmAlert(alertid_NoDBSelected);
    return;
  }
  PROP_propform_prepare(dbi);
  FrmGotoForm(formid_Properties);
}

static Boolean MAIN_hanevent(EventPtr event)
{
  Int16 ix;
  Boolean handled = FALSE;

  switch (event->eType)
  {

    case frmOpenEvent:
      MAIN_gMainform_init(FrmGetActiveForm());
      FrmDrawForm(MAIN_gMainform);
      MTAB_maintab_redraw();    /* just for the selection */
      handled = TRUE;
      break;

    case frmCloseEvent:
      MAIN_gMainform = NULL;
      MTAB_maintab_final();
      /* and pass on */
      break;

    case ctlRepeatEvent:
      if(event->data.ctlRepeat.controlID == btn_MainUp)
      {
        MTAB_maintab_scroll(FALSE, TRUE);
      }
      else if(event->data.ctlRepeat.controlID == btn_MainDn)
      {
        MTAB_maintab_scroll(TRUE, TRUE);
      }
      /* Repeating controls don't repeat if handled is set true. */
      break;

    case keyDownEvent:
      if(event->data.keyDown.chr == pageUpChr)
      {
        MTAB_maintab_scroll(FALSE, FALSE);
        handled = true;
      }
      else if(event->data.keyDown.chr == pageDownChr)
      {
        MTAB_maintab_scroll(TRUE, FALSE);
        handled = true;
      }
      break;

    case tblSelectEvent:
      if(event->data.tblSelect.tableID == table_Main)
      {
        Int32 dbi1, dbi2;
        UInt32 actual;

        MTAB_maintab_get_selection_dbi(&dbi1, NULL);

        MTAB_maintab_notice_selection(event->data.tblSelect.row,
                                      event->data.tblSelect.column);

        MTAB_maintab_get_selection_dbi(&dbi2, NULL);

        actual = TimGetSeconds();
        if( (actual <= (lastTap + 1) ) && ( dbi1 == dbi2 ) && ( dbi1 != -1 ) )
        {
          // doubletap -> View Info
          MAIN_drawProperties();
        }
        lastTap = actual;

        handled = TRUE;
      }
      break;

    case ctlSelectEvent:
      for(ix = 0; ix < Z_dNumColumns; ix++)
      {
        if(event->data.ctlSelect.controlID == SORT_gColumnState[ix].pop_resid)
        {
          short res;
          ListPtr ls;
          ls = FrmGetObjectPtr(MAIN_gMainform,
                            FrmGetObjectIndex(MAIN_gMainform,
                                              SORT_gColumnState[ix].
                                              list_resid));
          LstSetSelection(ls, SORT_gColumnState[ix].type);
          res = LstPopupList(ls);
          MTAB_do_column_header(ix, res);
          handled = TRUE;
          break;
        }
      }

      if(!handled)
      {
        switch (event->data.ctlSelect.controlID)
        {
          case pop_MainColPop:
          {
            short res;
            ListPtr ls;
            ls = FrmGetObjectPtr(MAIN_gMainform,
                                 FrmGetObjectIndex(MAIN_gMainform, ls_MainColPop));
            LstSetSelection(ls, -1);
            res = LstPopupList(ls);
            MTAB_do_name_header(res);
            handled = TRUE;
            break;
          }

          case chk_ShowRO:
            MTAB_do_filter_readonly(event->data.ctlSelect.on != FALSE);
            handled = TRUE;
            break;

          case btn_Unfilter:
            if(SORT_gFilter.active)
            {
              ControlPtr ctl;
              SORT_gFilter.active = FALSE;
              MTAB_do_refilter();
              ctl = FrmGetObjectPtr(MAIN_gMainform,
                                FrmGetObjectIndex(MAIN_gMainform,
                                                  btn_Unfilter));
              CtlHideControl(ctl);
            }
            handled = TRUE;
            break;

          case btn_Filter:
          {
            short col, filtype;
            long dbi;
            Z_DBInfo *ptr = NULL;

            MTAB_maintab_get_selection_dbi(&dbi, &col);
            if(dbi < 0 || dbi >= DBI_gNumDBs)
            {
              filtype = Z_dColumnTypeName;
              ptr = NULL;
            }
            else
            {
              if(col >= 1 && col < Z_dNumColumns + 1)
              {
                filtype = SORT_gColumnState[col - 1].type;
              }
              else
              {
                filtype = Z_dColumnTypeName;
              }
              if(filtype == Z_dColumnTypeAttributes)
              {
                filtype = Z_dColumnTypeName;
              }
              ptr = DBI_gDBList[dbi];
            }
            FILT_filterform_prepare(filtype, ptr);
            FrmPopupForm(formid_Filter);
            break;
          }
        }
      }
      break;


    case menuEvent:
      handled = MAIN_domenu(event->data.menu.itemID);
      break;
  }

  return handled;
}

static Boolean MAIN_domenu(UInt16 cmd)
{
  Boolean handled = false;
  Int32 dbi;
  Int16 res;
  Int16 showAlert;
  Z_DBInfo *db;
  Int32 res32;
  MemHandle resultHandle = NULL;
  UInt32 *resultForCreator;
  char errorText[100];

  switch (cmd)
  {
    case mitem_Properties:
      handled = TRUE;
      MAIN_drawProperties();
      break;

    case mitem_BeamOne:
      handled = TRUE;
      if(!MAIN_gOS3_avail)
      {
        FrmAlert(alertid_NoOS3Avail);
        break;
      }
      MTAB_maintab_get_selection_dbi(&dbi, NULL);
      if(dbi < 0 || dbi >= DBI_gNumDBs)
      {
        FrmAlert(alertid_NoDBSelected);
        break;
      }
      TOOL_beam_send_db(DBI_gDBList[dbi]);
      break;

    case mitem_Delete:
      handled = TRUE;
      MTAB_maintab_get_selection_dbi(&dbi, NULL);
      if(dbi < 0 || dbi >= DBI_gNumDBs)
      {
        FrmAlert(alertid_NoDBSelected);
        break;
      }
      db = DBI_gDBList[dbi];
      res = TOOL_delete_whole_db(db->card, db->lid, db->type, 1);
      if(!res)
      {
        break;
      }

      mRebuildAfterDelete;

      break;

    case mitem_DeleteFiltered:
      handled = TRUE;

      if(!MAIN_gBoneheadmode)
      {
        FrmAlert(alertid_NonBoneheadMode);
        break;
      }

      if(!SORT_gFilter.active)
      {
        FrmAlert(alertid_NoFilter);
        break;
      }

      if(DBI_gNumTabs < 1)
      {
        FrmAlert(alertid_NoDBSelected);
        break;
      }

      for(dbi = 0; dbi < DBI_gNumTabs; dbi++)
      {
        db = DBI_gDBList[DBI_gTabList[dbi]];
        showAlert = DBI_gNumTabs;
        // show alert just at the first time
        if(dbi > 0)
        {
          showAlert = 0;
        }
        res = TOOL_delete_whole_db(db->card, db->lid, db->type, showAlert);
        if( (showAlert > 0) && (!res) )
		{
		  break;
        }
      }

      mRebuildAfterDelete;

      break;

    case mitem_DeleteCreator:
      handled = TRUE;

      if(!MAIN_gBoneheadmode)
      {
        FrmAlert(alertid_NonBoneheadMode);
        break;
      }

      MTAB_maintab_get_selection_dbi(&dbi, NULL);
      if(dbi < 0 || dbi >= DBI_gNumDBs)
      {
        FrmAlert(alertid_NoDBSelected);
        break;
      }

      db = DBI_gDBList[dbi];

      /* Now allocate the space we need. */
      resultHandle = MemHandleNew(DBI_gNumDBs * sizeof(long));
      if(!resultHandle)
      {
        StrPrintF(errorText, "Error at allocating %d bytes -A- probably you have too less free memory",
                  DBI_gNumDBs * sizeof(long) );
        UTIL_printError( errorText );
        break;
      }

      resultForCreator = MemHandleLock(resultHandle);
      if(!resultForCreator)
      {
        UTIL_printError( "Error at MemHandleLock -MA-");
        MemHandleFree(resultHandle);
        resultHandle = NULL;
        break;
      }

      res32 = SORT_build_list_for_creator(db->creator, resultForCreator);

      for(dbi = 0; dbi < res32; dbi++)
      {
        db = DBI_gDBList[resultForCreator[dbi]];
        showAlert = res32;
        // show alert just at the first time
        if(dbi > 0)
        {
          showAlert = 0;
        }
        res = TOOL_delete_whole_db(db->card, db->lid, db->type, showAlert);
        if( (showAlert > 0) && (!res) )
		{
		  break;
        }
      }

      /* free memory */
      MemHandleFree(resultHandle);
      resultHandle = NULL;

      mRebuildAfterDelete;

      break;

    case mitem_CopyList:
    {
      MemPtr ptr;
      UInt32 lx, total = 0;

      for(dbi = 0; dbi < DBI_gNumTabs; dbi++)
      {
        total += StrLen(DBI_gDBList[DBI_gTabList[dbi]]->name);
        total++;
      }
      ptr = MemPtrNew(total);
      if(!ptr)
      {
        FrmAlert(alertid_NotEnoughMemory);
        break;
      }
      for(dbi = 0, lx = 0; dbi < DBI_gNumTabs; dbi++)
      {
        StrCopy(ptr + lx, DBI_gDBList[DBI_gTabList[dbi]]->name);
        lx += StrLen(DBI_gDBList[DBI_gTabList[dbi]]->name);

        /*
           causes compiler error
           ptr[lx] = '\n';
         */

        ((char *) ptr)[lx] = '\n';
        lx++;
      }
      ClipboardAddItem(clipboardText, ptr, total);
      MemPtrFree(ptr);
      break;
    }

    case mitem_ExportList:
    {
      char *title = "Database listing\n\n";
      Int16 titlelen = StrLen(title);
      UInt32 lx, total = 0;
      DmOpenRef memodb;
      UInt16 recix;
      MemHandle han;
      MemPtr ptr;

      total = titlelen + 1;

      for(dbi = 0; dbi < DBI_gNumTabs; dbi++)
      {
        total += StrLen(DBI_gDBList[DBI_gTabList[dbi]]->name);
        total++;
      }

      memodb = DmOpenDatabaseByTypeCreator('DATA', 'memo', dmModeReadWrite);
      if(memodb)
      {
        recix = 0;
        han = DmNewRecord(memodb, &recix, total);
        if(han)
        {
          ptr = MemHandleLock(han);
          if(ptr)
          {
            lx = 0;
            DmStrCopy(ptr, lx, title);
            lx += titlelen;

            for(dbi = 0; dbi < DBI_gNumTabs; dbi++)
            {
              DmStrCopy(ptr, lx, DBI_gDBList[DBI_gTabList[dbi]]->name);
              lx += StrLen(DBI_gDBList[DBI_gTabList[dbi]]->name);
              DmStrCopy(ptr, lx, "\n");
              lx++;
            }

            MemPtrUnlock(ptr);
          }
          else
          {
            UTIL_printError( "Error at MemHandleLock -MB-");
          }

          DmReleaseRecord(memodb, recix, TRUE);
        }
        else
        {
          StrPrintF(errorText, "DmNewRecord failed, error code %d.", DmGetLastErr() );
          UTIL_printError( errorText );
        }

        DmCloseDatabase(memodb);
      }
      else
      {
        StrPrintF(errorText, "DmOpenDatabaseByTypeCreator for memo failed, error code %d.", DmGetLastErr() );
        UTIL_printError( errorText );
      }

      break;
    }

    case mitem_ExportListCSV:
    {
      char *title = "Database CSV";
      Int16 titlelen = StrLen(title);
      char *name= "Name";
      UInt32 lx, total = 0;
      DmOpenRef memodb;
      UInt16 recix;
      MemHandle han;
      MemPtr ptr;
      Int16 i,j;
      char buffer[100];
      Int16 length;
      Int16 memosNeeded;
      Int32 memoLength;

      total = titlelen + 2 + 1; //\n\n and final \n

      total += StrLen( name );
      total++;  //;
      for( i = 0; i < Z_dNumColumnTypes; i++ )
      {
        total += StrLen( Z_cColumnName[i] );
        total++;  //;
      }
      total++; //EOL

      for(dbi = 0; dbi < DBI_gNumTabs; dbi++)
      {
        total += StrLen(DBI_gDBList[DBI_gTabList[dbi]]->name);
        total++;

        for( i = 0; i < Z_dNumColumnTypes; i++ )
        {
          MTAB_getString(DBI_gTabList[dbi], i, buffer, &length);
          total += length;
          total++; //;
        }
        total++; //EOL
      }

      memodb = DmOpenDatabaseByTypeCreator('DATA', 'memo', dmModeReadWrite);
      if(memodb)
      {
        recix = 0;

        //UTIL_printDebugInt(total, " bytes needed for CSV", NULL);

        //calculate number of needed memos
        memosNeeded = (total / dMaxInternalMemoLength ) + 1;

        if( memosNeeded > 1 )
        {
          memoLength = dMaxInternalMemoLength;
        }
        else
        {
          memoLength = total;
        }

        //UTIL_printDebugInt(memosNeeded, " memos needed for CSV", NULL);

        dbi = 0;
        for( j = 0; j < memosNeeded; j++ )
        {
          han = DmNewRecord(memodb, &recix, memoLength + 100 ); // hard to say what we need exactly
          if(han)
          {
            ptr = MemHandleLock(han);
            if(ptr)
            {
              lx = 0;
              // first line, will be title
              DmStrCopy(ptr, lx, title);
              lx += titlelen;
              //add title number if we have multiple memos
              if( memosNeeded > 1 )
              {
                StrIToA(buffer, j + 1);
                DmStrCopy(ptr, lx, buffer);
                lx += StrLen( buffer );
              }
              DmStrCopy(ptr, lx, "\n\n");
              lx += 2;
              if( j == 0 )
              {
				  // first line (CSV header : name)
				  DmStrCopy(ptr, lx, name );
				  lx += StrLen( name );
				  DmStrCopy(ptr, lx, ";");
				  lx++;
				  // first line (CSV header : all columns)
				  for( i = 0; i < Z_dNumColumnTypes; i++ )
				  {
					DmStrCopy(ptr, lx, Z_cColumnName[i] );
					lx += StrLen( Z_cColumnName[i] );
					DmStrCopy(ptr, lx, ";");
					lx++;
				  }
				  DmStrCopy(ptr, lx, "\n");
				  lx++;
		      }

              for( ; dbi < DBI_gNumTabs; dbi++)
              {
                DmStrCopy(ptr, lx, DBI_gDBList[DBI_gTabList[dbi]]->name);
                lx += StrLen(DBI_gDBList[DBI_gTabList[dbi]]->name);
                DmStrCopy(ptr, lx, ";");
                lx++;

                for( i = 0; i < Z_dNumColumnTypes; i++ )
                {
                  DmStrCopy(ptr, lx, MTAB_getString(DBI_gTabList[dbi], i, buffer, &length) );
                  lx += length;
                  DmStrCopy(ptr, lx, ";");
                  lx++;
                }

                DmStrCopy(ptr, lx, "\n");
                lx++;

                if( lx > memoLength )
                {
                  break;
                }
              }

              MemPtrUnlock(ptr);
            }
            else
            {
              UTIL_printError( "Error at MemHandleLock -MC-");
            }


            DmReleaseRecord(memodb, recix, TRUE);
            DmResizeRecord(memodb, recix, lx + 1);
            recix++;
            dbi++;
          }
          else
          {
            StrPrintF(errorText, "DmNewRecord failed, error code %d.", DmGetLastErr() );
            UTIL_printError( errorText );
          }
        }

        DmCloseDatabase(memodb);
      }
      else
      {
        StrPrintF(errorText, "DmOpenDatabaseByTypeCreator for memo failed, error code %d.", DmGetLastErr() );
        UTIL_printError( errorText );
      }

      break;
    }

    case mitem_ListCardInfo:
    {
      handled = TRUE;
      MenuEraseStatus(0);

      listFormShow = MAIN_eListFormCardInfo;

      FrmGotoForm(formid_ListForm);
      break;
    }

    case mitem_ListSystemInfo:
    {
      handled = TRUE;
      MenuEraseStatus(0);

      listFormShow = MAIN_eListFormSystemInfo;

      FrmGotoForm(formid_ListForm);
      break;
    }

    case mitem_Preferences:
      handled = TRUE;
      MenuEraseStatus(0);
      FrmPopupForm(formid_Preferences);
      break;

    case mitem_Info:
      handled = TRUE;
      MenuEraseStatus(0);
      FrmHelp(strid_AboutInfo);
      break;

    case mitem_Help:
      handled = TRUE;
      MenuEraseStatus(0);
      FrmHelp(strid_MainHelp);
      break;

    case mitem_About:
      handled = TRUE;
      MenuEraseStatus(0);
      {
        FormPtr frm = FrmInitForm(formid_About);
        Int16 res;

        while(1)
        {
          res = FrmDoDialog(frm);
          if(res == btn_AboutInfo)
          {
            FrmHelp(strid_AboutInfo);
          }
          else
          {
            break;
          }
        }

        FrmDeleteForm(frm);
      }
      break;

  }

  return handled;
}

void UpdateRepeatButtonsListForm()
{
  FieldPtr fieldPtr;
  UInt16 scrollPos;
  UInt16 textHeight;
  UInt16 fieldHeight;
  UInt16 upIndex;
  UInt16 downIndex;
  Boolean scrollUp;
  Boolean scrollDown;

  fieldPtr = FrmGetObjectPtr(listForm, FrmGetObjectIndex(listForm, fld_TextField));

  if(fieldPtr)
  {
    FldGetScrollValues (fieldPtr, &scrollPos, &textHeight,  &fieldHeight);

    scrollUp = false;
    scrollDown = false;

    if (textHeight > fieldHeight)
    {
      if (scrollPos > 0)
      {
        scrollUp = true;
      }

      if( scrollPos < ( textHeight - fieldHeight ) )
      {
        scrollDown = true;
      }
    }

    upIndex = FrmGetObjectIndex( listForm, btn_ListUp );
    downIndex = FrmGetObjectIndex( listForm, btn_ListDn );
    FrmUpdateScrollers( listForm, upIndex, downIndex, scrollUp, scrollDown );
  }
}

void PositionTextListForm(WinDirectionType direction)
{
  UInt16 lines;
  FieldPtr fieldPtr;

  fieldPtr = FrmGetObjectPtr(listForm, FrmGetObjectIndex(listForm, fld_TextField));

  if(fieldPtr)
  {
    lines = FldGetVisibleLines(fieldPtr);
    FldScrollField(fieldPtr, lines, direction);
  }

  UpdateRepeatButtonsListForm();
}

Boolean MAIN_formListForm_hanevent(EventPtr event)
{
  Boolean handled = FALSE;
  Int16 res;

  switch (event->eType)
  {
    case frmOpenEvent:
    {
      MemHandle h, oldHandle;
      char *text;
      FieldPtr fieldPtr;

      listForm = FrmGetActiveForm();
      fieldPtr = FrmGetObjectPtr(listForm, FrmGetObjectIndex(listForm, fld_TextField));

      FrmSetTitle(listForm, "Card Info" ) ;

      h = MemHandleNew(700);
      text = (char *) MemHandleLock( h );

      if( listFormShow == MAIN_eListFormCardInfo )
      {
        FrmSetTitle(listForm, "Card Info" ) ;
        TOOL_getCardInfo( text );
      }
      else
      if( listFormShow == MAIN_eListFormSystemInfo )
      {
        FrmSetTitle(listForm, "System Info" ) ;
        TOOL_getSystemInfo( text );
      }

      MemHandleResize( h, StrLen( text ) + 1 );
      MemHandleUnlock( h );
      oldHandle = FldGetTextHandle(fieldPtr);
      FldSetTextHandle(fieldPtr, h);
      if (oldHandle)
      {
        MemHandleFree(oldHandle);
      }
      FldRecalculateField (fieldPtr, true);
      FldDrawField (fieldPtr);

      UpdateRepeatButtonsListForm();

      FrmDrawForm( listForm );
      handled = TRUE;
      break;
    }

    case frmCloseEvent:
      listForm = NULL;
      /* and pass on */
      break;

    case ctlSelectEvent:
      switch (event->data.ctlSelect.controlID)
      {
        case btn_ListFormDone:
          FrmGotoForm(formid_Main);
          handled = TRUE;
          break;
      }
      break;

    case ctlRepeatEvent:
      if(event->data.ctlRepeat.controlID == btn_ListUp)
      {
        PositionTextListForm( winUp );
      }
      else if(event->data.ctlRepeat.controlID == btn_ListDn)
      {
        PositionTextListForm( winDown );
      }
      /* Repeating controls don't repeat if handled is set true. */
      break;

    case keyDownEvent:
      if(event->data.keyDown.chr == pageUpChr)
      {
        PositionTextListForm( winUp );
        handled = true;
      }
      else if(event->data.keyDown.chr == pageDownChr)
      {
        PositionTextListForm( winDown );
        handled = true;
      }
      break;

    default:
      break;
  }

  return handled;
}

