/*
   Tom'Catalog
   Copyright 2002, Thomas Aufischer

   based on:

   Z'Catalog: a PalmOS database utility
   Copyright 1998, Andrew Plotkin <erkyrath@eblong.com>
   http://www.eblong.com/zarf/zcatalog.html

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <PalmOS.h>
#include "zcatalog.h"
#include "resource.h"

/* Routines to manage the table in the top-level form (the list of
   databases.) */

static TablePtr tab = NULL;
static Int16 rows = 0;

static Int32 scrollat = 0;
static Int32 select_tab = -1;
static Int32 select_dbi = -1;
static Int16 select_column = 0;
static Boolean select_init = FALSE;
static char select_initname[dmDBNameLength];

void MTAB_drawcol_name(MemPtr tab, UInt16 row, UInt16 col,
                              RectanglePtr box);
void MTAB_drawcol_data(MemPtr tab, UInt16 row, UInt16 col,
                              RectanglePtr box);
void MTAB_adjust_scroll(void);

void MTAB_maintab_init()
{
  Int16 ix, jx;

  tab =
    FrmGetObjectPtr(MAIN_gMainform,
                    FrmGetObjectIndex(MAIN_gMainform, table_Main));

  rows = TblGetNumberOfRows(tab);

  for(ix = 0; ix < rows; ix++)
  {
    for(jx = 0; jx < Z_dNumColumns + 1; jx++)
    {
      TblSetItemStyle(tab, ix, jx, customTableItem);
    }
  }

  /* Set the column usable so that it draws and accepts user input. */
  for(jx = 0; jx < Z_dNumColumns + 1; jx++)
  {
    TblSetColumnUsable(tab, jx, TRUE);
  }

  TblSetCustomDrawProcedure(tab, 0, (TableDrawItemFuncPtr) MTAB_drawcol_name);
  for(jx = 0; jx < Z_dNumColumns; jx++)
  {
    TblSetCustomDrawProcedure(tab, 1 + jx,
                              (TableDrawItemFuncPtr) MTAB_drawcol_data);
  }
}

void MTAB_maintab_final()
{
  tab = NULL;
}

void MTAB_maintab_reload()
{
  Int16 ix;

  //UTIL_printDebugInt(DBI_gNumTabs, "DBI_gNumTabs", "MTAB_maintab_reload: before" );

  for(ix = 0; ix < rows; ix++)
  {

    Int32 dbi = ix + scrollat;

    //UTIL_printDebugInt(dbi, "dbi", "MTAB_maintab_reload: before" );

    if(dbi >= 0 && dbi < DBI_gNumTabs)
    {
      TblSetRowUsable(tab, ix, TRUE);
    }
    else
    {
      TblSetRowUsable(tab, ix, FALSE);
    }

  }

  //UTIL_printDebugInt(tab, "MTAB_maintab_reload: before Tbl", NULL);

  TblMarkTableInvalid(tab);

  /*{
     ControlPtr ctl;
     char *buf;
     ctl = FrmGetObjectPtr(MAIN_gMainform, FrmGetObjectIndex(MAIN_gMainform, lb_Count));
     buf = CtlGetLabel(ctl);
     StrIToA(buf, DBI_gNumTabs);
     StrCat(buf, " shown");
     CtlEraseControl(ctl);
     CtlSetLabel(ctl, buf);
     CtlDrawControl(ctl);
     } */

  //UTIL_printDebugInt(tab, "MTAB_maintab_reload: before adjust_scroll", NULL);

  MTAB_adjust_scroll();

  //UTIL_printDebug("MTAB_maintab_reload: before exit", NULL, NULL );

}

void MTAB_maintab_redraw()
{
  MTAB_maintab_redraw_careful( TRUE );
}

void MTAB_maintab_redraw_careful( Boolean unhighlight )
{
  if( unhighlight )
  {
    //UTIL_printDebugInt(tab, "MTAB_maintab_redraw_careful: before TBLUnhili", NULL);
    TblUnhighlightSelection(tab);
  }

  //UTIL_printDebugInt(tab, "MTAB_maintab_redraw_careful: before TBLRedraw", NULL);

  TblRedrawTable(tab);
  /* if (select_tab != -1 && select_tab >= scrollat && select_tab < scrollat+rows)
     {
       TblSelectItem(tab, select_tab-scrollat, select_column);
     }
  */

  //UTIL_printDebug("MTAB_maintab_reload: after MTAB_maintab_redraw_careful", NULL, NULL );
}

void MTAB_drawcol_name(MemPtr tab, UInt16 row, UInt16 col,
                              RectanglePtr box)
{
  FontID origfont;
  Int32 dbi;
  char *cx;
  Int16 width, len;
  Boolean fits;

  origfont = FntSetFont(stdFont);

  dbi = DBI_gTabList[(long) row + scrollat];
  cx = DBI_gDBList[dbi]->name;
  width = box->extent.x - 2;
  len = StrLen(cx);
  FntCharsInWidth(cx, &width, &len, &fits);

  WinDrawChars(cx, len, box->topLeft.x, box->topLeft.y);

  FntSetFont(origfont);
}

char* MTAB_getString(Int32 dbi, Int16 columnType, char *helpBuffer, Int16 *length)
{
  char *cx;
  Int16 len;
  Int16 help16;

  switch (columnType)
  {
    case Z_dColumnTypeType:
      cx = (char *) &(DBI_gDBList[dbi]->type);
      len = 4;
      break;

    case Z_dColumnTypeCreator:
      cx = (char *) &(DBI_gDBList[dbi]->creator);
      len = 4;
      break;

    case Z_dColumnTypeSize:
      cx = StrIToA(helpBuffer, DBI_gDBList[dbi]->size);
      len = StrLen(helpBuffer);
      if(len > 6)
      {
        help16 = ((DBI_gDBList[dbi]->size + 512) / 1024);
        StrPrintF(helpBuffer, "%dKB", help16);
        cx = helpBuffer;
        len = StrLen(helpBuffer);
      }
      break;

    case Z_dColumnTypeRecords:
      cx = StrIToA(helpBuffer, DBI_gDBList[dbi]->records);
      len = StrLen(helpBuffer);
      break;

    case Z_dColumnTypeAttributes:
      len = 0;
      if(DBI_gDBList[dbi]->attributes & dmHdrAttrResDB)
        helpBuffer[len++] = 'c';
      if(DBI_gDBList[dbi]->attributes & dmHdrAttrReadOnly)
        helpBuffer[len++] = 'r';
      if(DBI_gDBList[dbi]->attributes & dmHdrAttrAppInfoDirty)
        helpBuffer[len++] = 'a';
      if(DBI_gDBList[dbi]->attributes & dmHdrAttrBackup)
        helpBuffer[len++] = 'b';
      if(DBI_gDBList[dbi]->attributes & dmHdrAttrOKToInstallNewer)
        helpBuffer[len++] = 'n';
      if(DBI_gDBList[dbi]->attributes & dmHdrAttrResetAfterInstall)
        helpBuffer[len++] = 'i';
      if(DBI_gDBList[dbi]->attributes & dmHdrAttrCopyPrevention)
        helpBuffer[len++] = 'p';
      if(DBI_gDBList[dbi]->attributes & dmHdrAttrStream)
        helpBuffer[len++] = 's';
      if(DBI_gDBList[dbi]->attributes & dmHdrAttrOpen)
        helpBuffer[len++] = 'o';
      if(DBI_gDBList[dbi]->attributes & dmHdrAttrLaunchableData)
        helpBuffer[len++] = 'l';
      /* since Palm OS version 3.2 */
      if(DBI_gDBList[dbi]->attributes & dmHdrAttrHidden)
        helpBuffer[len++] = 'h';
      /* since Palm OS 4.0*/
      if (DBI_gDBList[dbi]->attributes & dmHdrAttrBundle)
         helpBuffer[len++] = 'u';
      /* since Palm OS 4.0*/
      if (DBI_gDBList[dbi]->attributes & dmHdrAttrRecyclable)
         helpBuffer[len++] = 'r';

      helpBuffer[len] = '\0';
      cx = helpBuffer;
      break;

    case Z_dColumnTypeCreated:
      cx = TOOL_dateStringShort(DBI_gDBList[dbi]->created, helpBuffer, &len);
      break;

    case Z_dColumnTypeModified:
      cx = TOOL_dateStringShort(DBI_gDBList[dbi]->modified, helpBuffer, &len);
      break;

    case Z_dColumnTypeBackuped:
      cx = TOOL_dateStringShort(DBI_gDBList[dbi]->backuped, helpBuffer, &len);
      break;

    case Z_dColumnTypeCreatedTime:
      cx = TOOL_timeStringShort(DBI_gDBList[dbi]->created, helpBuffer, &len);
      break;

    case Z_dColumnTypeModifiedTime:
      cx = TOOL_timeStringShort(DBI_gDBList[dbi]->modified, helpBuffer, &len);
      break;

    case Z_dColumnTypeBackupedTime:
      cx = TOOL_timeStringShort(DBI_gDBList[dbi]->backuped, helpBuffer, &len);
      break;

    case Z_dColumnTypeRWRO:
      cx = TOOL_getRWString( dbi, helpBuffer );
      len = StrLen(helpBuffer);
      break;

    case Z_dColumnTypeVersion:
      StrPrintF(helpBuffer, "%d.%ld",
      DBI_gDBList[dbi]->version,
      DBI_gDBList[dbi]->modifier );
      cx = helpBuffer;
      len = StrLen(helpBuffer);
      break;

    case Z_dColumnTypeCard:
      StrPrintF(helpBuffer, "%d:%s",
      DBI_gDBList[dbi]->card,
      TOOL_locationString( DBI_gDBList[dbi]->card, DBI_gDBList[dbi]->lid ) );
      cx = helpBuffer;
      len = StrLen(helpBuffer);
      break;

    default:
      cx = "???";
      len = 3;
      break;
  }

  *length = len;
  StrCopy( helpBuffer, cx );
  return helpBuffer;
}

void MTAB_drawcol_data(MemPtr tab, UInt16 row, UInt16 col,
                              RectanglePtr box)
{
  FontID origfont;
  Int32 dbi;
  char *cx;
  char buffer[100];
  Int16 width, len;
  Boolean fits;

  origfont = FntSetFont(stdFont);

  dbi = DBI_gTabList[(long) row + scrollat];

  cx = MTAB_getString( dbi, SORT_gColumnState[col - 1].type, buffer, &len );

  width = box->extent.x - 1;
  FntCharsInWidth(cx, &width, &len, &fits);

  WinDrawChars(cx, len, box->topLeft.x, box->topLeft.y);

  FntSetFont(origfont);
}

void MTAB_do_sort(Int16 type, Int16 reverse)
{
  if((SORT_gSortType != type) || (SORT_gSortReverse != reverse))
  {
    SORT_gSortType = type;
    SORT_gSortReverse = reverse;
    MTAB_do_refilter();
  }
}

void MTAB_do_name_header(Int16 newval)
{
  if(newval == 0)
  {
    MTAB_do_sort(Z_dColumnTypeName, 0);
  }
  else
  {
    MTAB_do_sort(Z_dColumnTypeName, 1);
  }
}

void MTAB_do_column_header(Int16 dcol, Int16 newval)
{
  Int16 jx, oldval;

  if(dcol < 0 || dcol >= Z_dNumColumns)
  {
    return;
  }

  if(newval == 0)
  {
    /* sort by column */
    MTAB_do_sort(SORT_gColumnState[dcol].type, 0);
  }
  else if(newval == 1)
  {
    /* sort by column revese */
    MTAB_do_sort(SORT_gColumnState[dcol].type, 1);
  }
  else if(newval >= 2 && newval < (Z_dNumColumnTypes + 2))
  {
    newval -= 2;

    oldval = SORT_gColumnState[dcol].type;
    if(oldval == newval)
    {
      return;
    }
    SORT_gColumnState[dcol].type = newval;
    MTAB_change_inval_column(dcol);

    for(jx = 0; jx < Z_dNumColumns; jx++)
    {
      if(jx != dcol && SORT_gColumnState[jx].type == newval)
      {
        SORT_gColumnState[jx].type = oldval;
        MTAB_change_inval_column(jx);
        break;
      }
    }

    MTAB_maintab_redraw();
  }
}

void MTAB_do_filter_readonly(Boolean newval)
{
  if(newval != SORT_gShowReadonly)
  {
    SORT_gShowReadonly = newval;
    MTAB_do_refilter();
  }
}

void MTAB_do_refilter()
{
  /* Fill in the DBI_gTabList data again. */
  SORT_build_tabList();
  select_tab = -1;
  MTAB_sanitize_selscroll();
  MTAB_maintab_reload();
  MTAB_maintab_redraw();
}

void MTAB_change_inval_column(Int16 col)
{
  ControlPtr ctl;

  ctl = FrmGetObjectPtr(MAIN_gMainform,
                        FrmGetObjectIndex(MAIN_gMainform,
                                          SORT_gColumnState[col].pop_resid));

  CtlSetLabel(ctl, Z_cColumnName[SORT_gColumnState[col].type]);

  TblMarkTableInvalid(tab);
}

void MTAB_sanitize_selscroll()
{
  if(select_dbi < 0 || select_dbi >= DBI_gNumDBs)
  {
    select_dbi = -1;
  }

  if(select_tab < 0 || select_tab >= DBI_gNumTabs)
  {
    select_tab = -1;
  }

  if(select_dbi == -1 && select_init)
  {
    select_dbi = DBI_find_dbi_by_name(select_initname);
    select_init = FALSE;
  }

  if(select_dbi == -1)
  {
    select_tab = -1;
  }
  else
  {
    if(select_tab == -1)
    {
      Int32 ix;

      /* Find the selected dbi. */
      for(ix = 0; ix < DBI_gNumTabs; ix++)
      {
        if(DBI_gTabList[ix] == select_dbi)
        {
          break;
        }
      }
      if(ix >= DBI_gNumTabs)
      {
        /* Not there. Forget it. */
        select_dbi = -1;
      }
      else
      {
        select_tab = ix;
      }
    }
  }

  if(select_tab >= 0)
  {
    /* Fetch the selected item onto the screen. */

    if(select_tab >= scrollat && select_tab < scrollat + rows)
    {
      /* Oh, it already is. */
    }
    else
    {
      scrollat = select_tab - ((rows - 1) / 2);
    }
  }

  if(scrollat > DBI_gNumTabs - rows)
  {
    scrollat = DBI_gNumTabs - rows;
  }
  if(scrollat < 0)
  {
    scrollat = 0;
  }

  //UTIL_printDebugInt(DBI_gNumTabs, "sanitize_selscroll: after", "DBI_gNumTabs");
  //UTIL_printDebugInt(scrollat, "sanitize_selscroll: after", "scrollat");
}

void MTAB_maintab_scroll(Boolean down, Boolean oneline)
{
  Int16 ix;
  Int32 newscrollat;

  ix = (oneline) ? 1 : rows;
  if(!down)
  {
    ix = -ix;
  }

  newscrollat = scrollat + ix;

  if(newscrollat > DBI_gNumTabs - (rows))
  {
    newscrollat = DBI_gNumTabs - (rows);
  }
  if(newscrollat < 0)
  {
    newscrollat = 0;
  }

  if(newscrollat != scrollat)
  {
    scrollat = newscrollat;
    MTAB_maintab_reload();

    MTAB_maintab_redraw();
  }
}

void MTAB_adjust_scroll()
{
  Int16 upindex;
  Int16 downindex;

  upindex = FrmGetObjectIndex(MAIN_gMainform, btn_MainUp);
  downindex = FrmGetObjectIndex(MAIN_gMainform, btn_MainDn);

  //UTIL_printDebug("adjust_scroll: before FrmUpdate", NULL, NULL );

  FrmUpdateScrollers(MAIN_gMainform, upindex, downindex,
                     (scrollat > 0), (scrollat < DBI_gNumTabs - (rows)));


  //UTIL_printDebug("MTAB_maintab_reload: after FrmUpdate", NULL, NULL );

}

void MTAB_maintab_set_init_selection(Boolean isselect, char *name, Int16 col)
{
  select_init = isselect;
  StrCopy(select_initname, name);
  select_column = col;
}

void MTAB_maintab_notice_selection(Int16 row, Int16 col)
{
  if(row < 0)
  {
    select_tab = -1;
    select_dbi = -1;
  }
  else
  {
    select_column = col;
    select_tab = row + scrollat;
    select_dbi = DBI_gTabList[select_tab];
  }
}

void MTAB_maintab_get_selection_dbi(Int32 * dbi, Int16 * col)
{
  *dbi = select_dbi;
  if(col)
  {
    *col = select_column;
  }
}
