/*
   Tom'Catalog
   Copyright 2002, Thomas Aufischer

   based on:

   Z'Catalog: a PalmOS database utility
   Copyright 1998, Andrew Plotkin <erkyrath@eblong.com>
   http://www.eblong.com/zarf/zcatalog.html

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <PalmOS.h>
#include "zcatalog.h"
#include "resource.h"

/* The form handler for the Properties form. */

static FormPtr propform = NULL;

static Int32 orig_dbi;
static UInt16 card;
static LocalID lid;

static struct
{
  char name[dmDBNameLength];
  Boolean inrom;
  UInt16 attr;
  UInt16 version;
  Z_DateTime create, mod, backup;
  /* UInt32 modnum; */
  LocalID appinfo_id;
  LocalID sortinfo_id;
  UInt32 type;
  UInt32 creator;
  UInt32 records;
  UInt32 fullsize;
}
work;

#define mGetObject(id)  \
    (FrmGetObjectPtr(propform, FrmGetObjectIndex(propform, (id))))

static void PROP_setup_field(Int16 resid, char *str);
static Int16 PROP_unpack_test_changed(Z_Changedata * newdat);

void PROP_propform_prepare(Int32 dbi)
{
  Z_DBInfo *db = DBI_gDBList[dbi];
  orig_dbi = dbi;
  card = db->card;
  lid = db->lid;
}

static Boolean PROP_propform_init(FormPtr frm)
{
  Err err;
  char idbuf[dmDBNameLength], helpBuffer[dmDBNameLength];
  ControlPtr ctl;

  propform = frm;

  /* Read in all sorts of info about the DB. */

  err = DmDatabaseInfo(card, lid, work.name,
                       &work.attr, &work.version,
                       &work.create.base, &work.mod.base, &work.backup.base,
                       NULL, &work.appinfo_id, &work.sortinfo_id,
                       &work.type, &work.creator);

  if(err)
  {
    return FALSE;
  }

  err = DmDatabaseSize(card, lid, &work.records, &work.fullsize, NULL);
  if(err)
  {
    return FALSE;
  }

  work.inrom = ( TOOL_location(card, lid) == TOOL_eLocation_ROM );

  /* Set all the form fields appropriately. */

  PROP_setup_field(fld_PropName, work.name);
  Z_mOSTypeToBuf(idbuf, work.type);
  PROP_setup_field(fld_PropType, idbuf);
  Z_mOSTypeToBuf(idbuf, work.creator);
  PROP_setup_field(fld_PropCreator, idbuf);

  StrCopy(idbuf, "Card ");
  StrIToA(helpBuffer, card);
  StrCat(idbuf, helpBuffer);
  StrCat(idbuf, ", ID ");
  StrIToA(helpBuffer, lid);
  StrCat(idbuf, helpBuffer);
  PROP_setup_field(fld_PropCardLid, idbuf);

  TOOL_fill_out_temp(&work.create);
  TOOL_fill_out_temp(&work.mod);
  TOOL_fill_out_temp(&work.backup);

  TOOL_dateTimeString(&work.create);
  TOOL_dateTimeString(&work.mod);
  TOOL_dateTimeString(&work.backup);

  ctl = mGetObject(sel_PropCreateDate);
  CtlSetLabel(ctl, work.create.date_buf);
  ctl = mGetObject(sel_PropCreateTime);
  CtlSetLabel(ctl, work.create.time_buf);

  ctl = mGetObject(sel_PropModDate);
  CtlSetLabel(ctl, work.mod.date_buf);
  ctl = mGetObject(sel_PropModTime);
  CtlSetLabel(ctl, work.mod.time_buf);

  ctl = mGetObject(sel_PropBackupDate);
  CtlSetLabel(ctl, work.backup.date_buf);
  ctl = mGetObject(sel_PropBackupTime);
  CtlSetLabel(ctl, work.backup.time_buf);

  StrCopy(idbuf, TOOL_locationString(card, lid));
  StrCat(idbuf, ": ");
  StrIToA(helpBuffer, work.records);
  StrCat(idbuf, helpBuffer);
  if(work.records == 1)
    StrCat(idbuf, " record, ");
  else
    StrCat(idbuf, " records, ");
  StrIToA(helpBuffer, work.fullsize);
  StrCat(idbuf, helpBuffer);
  StrCat(idbuf, " bytes");
  PROP_setup_field(fld_PropRecordsSize, idbuf);

  ctl = mGetObject(chk_PropResource);
  CtlSetValue(ctl, (work.attr & dmHdrAttrResDB) != 0);
  ctl = mGetObject(chk_PropReadOnly);
  CtlSetValue(ctl, (work.attr & dmHdrAttrReadOnly) != 0);
  ctl = mGetObject(chk_PropAppInfoDirty);
  CtlSetValue(ctl, (work.attr & dmHdrAttrAppInfoDirty) != 0);
  ctl = mGetObject(chk_PropBackup);
  CtlSetValue(ctl, (work.attr & dmHdrAttrBackup) != 0);
  ctl = mGetObject(chk_PropInstallNewer);
  CtlSetValue(ctl, (work.attr & dmHdrAttrOKToInstallNewer) != 0);
  ctl = mGetObject(chk_PropResetInstall);
  CtlSetValue(ctl, (work.attr & dmHdrAttrResetAfterInstall) != 0);
  ctl = mGetObject(chk_PropCopyPrev);
  CtlSetValue(ctl, (work.attr & dmHdrAttrCopyPrevention) != 0);
  ctl = mGetObject(chk_PropStream);
  CtlSetValue(ctl, (work.attr & dmHdrAttrStream) != 0);
  ctl = mGetObject(chk_PropOpen);
  CtlSetValue(ctl, (work.attr & dmHdrAttrOpen) != 0);

  if(!MAIN_gBoneheadmode || work.inrom)
  {
    FieldPtr fld;
    FieldAttrType fldattr;

    ctl = mGetObject(btn_PropCancel);
    CtlSetUsable(ctl, FALSE);
    ctl = mGetObject(btn_PropDelete);
    CtlSetUsable(ctl, FALSE);

    ctl = mGetObject(chk_PropResource);
    CtlSetEnabled(ctl, FALSE);
    ctl = mGetObject(chk_PropReadOnly);
    CtlSetEnabled(ctl, FALSE);
    ctl = mGetObject(chk_PropAppInfoDirty);
    CtlSetEnabled(ctl, FALSE);
    ctl = mGetObject(chk_PropBackup);
    CtlSetEnabled(ctl, FALSE);
    ctl = mGetObject(chk_PropInstallNewer);
    CtlSetEnabled(ctl, FALSE);
    ctl = mGetObject(chk_PropResetInstall);
    CtlSetEnabled(ctl, FALSE);
    ctl = mGetObject(chk_PropCopyPrev);
    CtlSetEnabled(ctl, FALSE);
    ctl = mGetObject(chk_PropStream);
    CtlSetEnabled(ctl, FALSE);
    ctl = mGetObject(chk_PropOpen);
    CtlSetEnabled(ctl, FALSE);

    ctl = mGetObject(sel_PropCreateDate);
    CtlSetEnabled(ctl, FALSE);
    ctl = mGetObject(sel_PropModDate);
    CtlSetEnabled(ctl, FALSE);
    ctl = mGetObject(sel_PropBackupDate);
    CtlSetEnabled(ctl, FALSE);
    ctl = mGetObject(sel_PropCreateTime);
    CtlSetEnabled(ctl, FALSE);
    ctl = mGetObject(sel_PropModTime);
    CtlSetEnabled(ctl, FALSE);
    ctl = mGetObject(sel_PropBackupTime);
    CtlSetEnabled(ctl, FALSE);

    fld = mGetObject(fld_PropName);
    FldGetAttributes(fld, &fldattr);
    fldattr.editable = FALSE;
    FldSetAttributes(fld, &fldattr);

    fld = mGetObject(fld_PropType);
    FldGetAttributes(fld, &fldattr);
    fldattr.editable = FALSE;
    FldSetAttributes(fld, &fldattr);

    fld = mGetObject(fld_PropCreator);
    FldGetAttributes(fld, &fldattr);
    fldattr.editable = FALSE;
    FldSetAttributes(fld, &fldattr);
  }

  return TRUE;
}

static void PROP_setup_field(Int16 resid, char *str)
{
  if(StrLen(str))
  {
    FieldPtr fld;
    MemHandle han, oldhan;
    char *ptr;

    fld = mGetObject(resid);
    oldhan = FldGetTextHandle(fld); /* probably NULL, but anyhow */

    han = (MemHandle) MemHandleNew(StrLen(str) + 1);
    if(han)
    {
      ptr = MemHandleLock((MemHandle) han);
      if(ptr)
      {
        StrCopy(ptr, str);
        MemHandleUnlock((MemHandle) han);
        FldSetTextHandle(fld, han);
        if(oldhan)
        {
          MemHandleFree((MemHandle) oldhan);
        }
      }
    }
  }
}

Boolean PROP_propform_unload_state(Z_Preferences * prefs)
{
  if(!propform)
  {
    return FALSE;
  }

  prefs->viewstate = 1;
  StrCopy(prefs->viewname, work.name);

  return TRUE;
}

Boolean PROP_propform_hanevent(EventPtr event)
{
  Boolean handled = FALSE;
  Int16 res;
  Z_Changedata newdat;

  switch (event->eType)
  {
    case frmOpenEvent:
      if(!PROP_propform_init(FrmGetActiveForm()))
      {
        FrmAlert(alertid_CannotReadDB);
        FrmGotoForm(formid_Main);
        handled = TRUE;
        break;
      }
      FrmDrawForm(propform);
      handled = TRUE;
      break;

    case frmCloseEvent:
      propform = NULL;
      /* and pass on */
      break;

    case menuEvent:
      if(event->data.menu.itemID == mitem_PropHelp)
      {
        MenuEraseStatus(0);
        if(MAIN_gBoneheadmode)
        {
          if(work.inrom)
          {
            FrmHelp(strid_PropertiesEditHelp);  /*### */
          }
          else
          {
            FrmHelp(strid_PropertiesEditHelp);
          }
        }
        else
        {
          FrmHelp(strid_PropertiesViewHelp);
        }
      }
      handled = TRUE;
      break;


    case ctlSelectEvent:
      switch (event->data.ctlSelect.controlID)
      {
        case sel_PropCreateDate:
        case sel_PropModDate:
        case sel_PropBackupDate:
        {
          ControlPtr ctl;
          Z_DateTime *tem = NULL;
          char *label = NULL;
          Int16 datectl = 0, timectl = 0;

          switch (event->data.ctlSelect.controlID)
          {
            case sel_PropCreateDate:
              tem = &work.create;
              datectl = sel_PropCreateDate;
              timectl = sel_PropCreateTime;
              label = "DB Creation Date";
              break;

            case sel_PropModDate:
              tem = &work.mod;
              datectl = sel_PropModDate;
              timectl = sel_PropModTime;
              label = "DB Modification Date";
              break;

            case sel_PropBackupDate:
              tem = &work.backup;
              datectl = sel_PropBackupDate;
              timectl = sel_PropBackupTime;
              label = "DB Backup Date";
              break;
          }

          if(!tem->active)
          {
            TimSecondsToDateTime(TimGetSeconds(), &tem->fields);
          }
          res = SelectDay(selectDayByDay, &tem->fields.month, &tem->fields.day,
                          &tem->fields.year, label);
          if(res)
          {
            tem->active = TRUE;
            TOOL_dateTimeString(tem);
            ctl = mGetObject(datectl);
            CtlSetLabel(ctl, tem->date_buf);
            ctl = mGetObject(timectl);
            CtlSetLabel(ctl, tem->time_buf);
          }
          handled = TRUE;
          break;
        }

        case sel_PropCreateTime:
        case sel_PropModTime:
        case sel_PropBackupTime:
          FrmAlert(alertid_TimeNotImplemented);
          break;

        case chk_PropResource:
          res = ((work.attr & dmHdrAttrResDB) != 0);
          if(event->data.ctlSelect.on)
          {
            res = !res;
          }
          if(res)
          {
            res = FrmAlert(alertid_ReallyChangeResourceFlag);
            if(res == 1)
            {
              CtlSetValue(event->data.ctlSelect.pControl,
                          ((work.attr & dmHdrAttrResDB) != 0));
            }
          }
          handled = TRUE;
          break;

        case chk_PropCopyPrev:
          res = ((work.attr & dmHdrAttrCopyPrevention) != 0);
          if(event->data.ctlSelect.on)
          {
            res = FALSE;        /* only complain about turning it on */
          }
          if(res)
          {
            res = FrmAlert(alertid_ReallyChangeCopyPrevFlag);
            if(res == 1)
            {
              CtlSetValue(event->data.ctlSelect.pControl,
                          ((work.attr & dmHdrAttrCopyPrevention) != 0));
            }
          }
          handled = TRUE;
          break;

        case chk_PropOpen:
          res = ((work.attr & dmHdrAttrOpen) != 0);
          if(event->data.ctlSelect.on)
          {
            res = !res;
          }
          if(res)
          {
            res = FrmAlert(alertid_ReallyChangeOpenFlag);
            if(res == 1)
            {
              CtlSetValue(event->data.ctlSelect.pControl,
                          ((work.attr & dmHdrAttrOpen) != 0));
            }
          }
          handled = TRUE;
          break;

        case btn_PropDone:
          if(MAIN_gBoneheadmode)
          {
            res = PROP_unpack_test_changed(&newdat);
            if(res)
            {
              if(!TOOL_change_prop_db(res, &newdat, card, lid))
              {
                handled = TRUE;
                break;          /* cancelled or error */
              }
            }
          }
          FrmGotoForm(formid_Main);
          handled = TRUE;
          break;

        case btn_PropCancel:
          FrmGotoForm(formid_Main);
          handled = TRUE;
          break;

        case btn_PropDelete:
          res = TOOL_delete_whole_db(card, lid, work.type, 1);
          if(!res)
          {
            /* cancelled or error */
            handled = TRUE;
            break;
          }
          FrmGotoForm(formid_Main);
          handled = TRUE;
          break;
      }
      break;
  }

  return handled;
}

/* Read all the form fields back into a Z_Changedata struct. Compare
   those values with the original values (in the work struct), and
   return a bitfield saying which fields have changed. If the return
   value is zero, nothing has changed. */
static Int16 PROP_unpack_test_changed(Z_Changedata * newdat)
{
  Int16 ix;
  FieldPtr fld;
  ControlPtr ctl;
  MemHandle han = NULL;
  MemPtr ptr = NULL;
  Int16 len = 0;
  Int16 res = 0;
  Int16 gotdata = 0;
  UInt32 uval;

  /* Check the attribute checkboxes */
  newdat->attributes = work.attr;

  ctl = mGetObject(chk_PropResource);
  if(CtlGetValue(ctl))
    newdat->attributes |= dmHdrAttrResDB;
  else
    newdat->attributes &= ~dmHdrAttrResDB;

  ctl = mGetObject(chk_PropReadOnly);
  if(CtlGetValue(ctl))
    newdat->attributes |= dmHdrAttrReadOnly;
  else
    newdat->attributes &= ~dmHdrAttrReadOnly;

  ctl = mGetObject(chk_PropAppInfoDirty);
  if(CtlGetValue(ctl))
    newdat->attributes |= dmHdrAttrAppInfoDirty;
  else
    newdat->attributes &= ~dmHdrAttrAppInfoDirty;

  ctl = mGetObject(chk_PropBackup);
  if(CtlGetValue(ctl))
    newdat->attributes |= dmHdrAttrBackup;
  else
    newdat->attributes &= ~dmHdrAttrBackup;

  ctl = mGetObject(chk_PropInstallNewer);
  if(CtlGetValue(ctl))
    newdat->attributes |= dmHdrAttrOKToInstallNewer;
  else
    newdat->attributes &= ~dmHdrAttrOKToInstallNewer;

  ctl = mGetObject(chk_PropResetInstall);
  if(CtlGetValue(ctl))
    newdat->attributes |= dmHdrAttrResetAfterInstall;
  else
    newdat->attributes &= ~dmHdrAttrResetAfterInstall;

  ctl = mGetObject(chk_PropCopyPrev);
  if(CtlGetValue(ctl))
    newdat->attributes |= dmHdrAttrCopyPrevention;
  else
    newdat->attributes &= ~dmHdrAttrCopyPrevention;

  ctl = mGetObject(chk_PropStream);
  if(CtlGetValue(ctl))
    newdat->attributes |= dmHdrAttrStream;
  else
    newdat->attributes &= ~dmHdrAttrStream;

  ctl = mGetObject(chk_PropOpen);
  if(CtlGetValue(ctl))
    newdat->attributes |= dmHdrAttrOpen;
  else
    newdat->attributes &= ~dmHdrAttrOpen;

  gotdata |= Z_dChangedAttr;

  /* Check the name field */
  fld = mGetObject(fld_PropName);
  han = FldGetTextHandle(fld);
  if(han)
  {
    ptr = MemHandleLock((MemHandle) han);
    len = FldGetTextLength(fld);
    if(ptr)
    {
      StrNCopy(newdat->name, ptr, 31);
      newdat->name[31] = '\0';
      gotdata |= Z_dChangedName;
    }
    MemHandleUnlock((MemHandle) han);
  }

  /* Check the type field */
  fld = mGetObject(fld_PropType);
  han = FldGetTextHandle(fld);
  if(han)
  {
    ptr = MemHandleLock((MemHandle) han);
    len = FldGetTextLength(fld);
    if(ptr)
    {
      newdat->type = 0;
      for(ix = 0; ix < 4; ix++)
      {
        char ch;
        if(ix < len)
        {
          /*
             causes compiler error
             ch = ptr[ix];
           */

          ch = ((char *) ptr)[ix];
        }
        else
        {
          ch = ' ';
        }
        newdat->type = (newdat->type << 8) | (ch & 0xff);
      }
      gotdata |= Z_dChangedType;
    }
    MemHandleUnlock((MemHandle) han);
  }

  /* Check the creator field */
  fld = mGetObject(fld_PropCreator);
  han = FldGetTextHandle(fld);
  if(han)
  {
    ptr = MemHandleLock((MemHandle) han);
    len = FldGetTextLength(fld);
    if(ptr)
    {
      newdat->creator = 0;
      for(ix = 0; ix < 4; ix++)
      {
        char ch;
        if(ix < len)
        {
          /*
             causes compiler error
             ch = ptr[ix];
           */

          ch = ((char *) ptr)[ix];
        }
        else
        {
          ch = ' ';
        }
        newdat->creator = (newdat->creator << 8) | (ch & 0xff);
      }
      gotdata |= Z_dChangedCreator;
    }
    MemHandleUnlock((MemHandle) han);
  }

  /* Check the timestamps. */
  newdat->create_date = work.create.base;
  newdat->mod_date = work.mod.base;
  newdat->backup_date = work.backup.base;

  uval = TOOL_temp_to_base(&work.create);
  if(uval != work.create.base)
  {
    newdat->create_date = uval;
    gotdata |= Z_dChangedTimes;
  }

  uval = TOOL_temp_to_base(&work.mod);
  if(uval != work.mod.base)
  {
    newdat->mod_date = uval;
    gotdata |= Z_dChangedTimes;
  }

  uval = TOOL_temp_to_base(&work.backup);
  if(uval != work.backup.base)
  {
    newdat->backup_date = uval;
    gotdata |= Z_dChangedTimes;
  }

  /* Put it all together. */

  res = 0;

  if(gotdata & Z_dChangedName)
  {
    if(StrCompare(newdat->name, work.name))
    {
      res |= Z_dChangedName;
    }
  }

  if(gotdata & Z_dChangedType)
  {
    if(newdat->type != work.type)
    {
      res |= Z_dChangedType;
    }
  }

  if(gotdata & Z_dChangedCreator)
  {
    if(newdat->creator != work.creator)
    {
      res |= Z_dChangedCreator;
    }
  }

  if(gotdata & Z_dChangedAttr)
  {
    if(newdat->attributes != work.attr)
    {
      res |= Z_dChangedAttr;
    }
  }

  if(gotdata & Z_dChangedTimes)
  {
    res |= Z_dChangedTimes;
  }

  return res;
}
