/*
   Tom'Catalog
   Copyright 2002, Thomas Aufischer

   based on:

   Z'Catalog: a PalmOS database utility
   Copyright 1998, Andrew Plotkin <erkyrath@eblong.com>
   http://www.eblong.com/zarf/zcatalog.html

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <PalmOS.h>
#include "zcatalog.h"
#include "resource.h"


Int16 SORT_gSortType;
Int16 SORT_gSortReverse;
Int16 SORT_gShowReadonly;

Z_Filter SORT_gFilter;
Z_Column SORT_gColumnState[Z_dNumColumns];

void SORT_set_default_column_state()
{
  Int16 ix;

  SORT_gSortType = Z_dColumnTypeName;
  for(ix = 0; ix < Z_dNumColumns; ix++)
  {
    SORT_gColumnState[ix].type = ix;
  }

  SORT_gColumnState[0].list_resid = ls_MainColPop0;
  SORT_gColumnState[1].list_resid = ls_MainColPop1;
  SORT_gColumnState[2].list_resid = ls_MainColPop2;
  SORT_gColumnState[0].pop_resid = pop_MainColPop0;
  SORT_gColumnState[1].pop_resid = pop_MainColPop1;
  SORT_gColumnState[2].pop_resid = pop_MainColPop2;

  MemSet(&SORT_gFilter, sizeof(Z_Filter), 0);
  SORT_gFilter.active = FALSE;
}

void SORT_load_pref_column_state(Z_Preferences * pref)
{
  Int16 ix;

  SORT_gSortType = pref->sortType;
  SORT_gShowReadonly = pref->showReadonly;
  SORT_gSortReverse = pref->sortReverse;
  SORT_gFilter= pref->filter;

  for(ix = 0; ix < Z_dNumColumns; ix++)
  {
    SORT_gColumnState[ix].type = pref->coltype[ix];
  }

  MTAB_maintab_set_init_selection(pref->isselect, pref->selectname,
                                  pref->selectcol);
}

void SORT_unload_pref_column_state(Z_Preferences * pref)
{
  Int16 ix;
  Int32 dbi;
  Int16 col;

  MTAB_maintab_get_selection_dbi(&dbi, &col);

  pref->sortType = SORT_gSortType;
  pref->showReadonly = SORT_gShowReadonly;
  pref->sortReverse = SORT_gSortReverse;
  pref->filter = SORT_gFilter;

  for(ix = 0; ix < Z_dNumColumns; ix++)
  {
    pref->coltype[ix] = SORT_gColumnState[ix].type;
  }

  if(dbi >= 0 && dbi < DBI_gNumDBs)
  {
    pref->isselect = TRUE;
    StrCopy(pref->selectname, DBI_gDBList[dbi]->name);
  }
  else
  {
    pref->isselect = FALSE;
    pref->selectname[0] = '\0';
  }
  pref->selectcol = col;
}

static Int16 SORT_sorter_name(void *d1, void *d2, Int32 rock)
{
  Int32 dl1, dl2;
  Z_DBInfo *p1, *p2;
  Int16 res;

  dl1 = *((Int32 *) d1);
  dl2 = *((Int32 *) d2);
  p1 = DBI_gDBList[dl1];
  p2 = DBI_gDBList[dl2];

  res = StrCaselessCompare(p1->name, p2->name);

  if(SORT_gSortReverse)
  {
    res = res * (-1);
  }

  return res;
}

static Int16 SORT_sorter_type(void *d1, void *d2, Int32 rock)
{
  Int32 dl1, dl2;
  Z_DBInfo *p1, *p2;
  Int16 res;
  char buf1[dmDBNameLength], buf2[dmDBNameLength];

  dl1 = *((Int32 *) d1);
  dl2 = *((Int32 *) d2);
  p1 = DBI_gDBList[dl1];
  p2 = DBI_gDBList[dl2];


  Z_mOSTypeToBuf(buf1, p1->type);
  Z_mOSTypeToBuf(buf2, p2->type);

  res = StrCaselessCompare( buf1, buf2 );
  if( res == 0 )
  {
    res = StrCaselessCompare(p1->name, p2->name);
  }

  if(SORT_gSortReverse)
  {
    res = res * (-1);
  }

  return res;
}

static Int16 SORT_sorter_creator(void *d1, void *d2, Int32 rock)
{
  Int32 dl1, dl2;
  Z_DBInfo *p1, *p2;
  Int16 res;
  char buf1[dmDBNameLength], buf2[dmDBNameLength];

  dl1 = *((Int32 *) d1);
  dl2 = *((Int32 *) d2);
  p1 = DBI_gDBList[dl1];
  p2 = DBI_gDBList[dl2];

  Z_mOSTypeToBuf(buf1, p1->creator );
  Z_mOSTypeToBuf(buf2, p2->creator );

  res = StrCaselessCompare( buf1, buf2 );
  if( res == 0 )
  {
    res = StrCaselessCompare(p1->name, p2->name);
  }

  if(SORT_gSortReverse)
  {
    res = res * (-1);
  }

  return res;
}

static Int16 SORT_sorter_size(void *d1, void *d2, Int32 rock)
{
  Int32 dl1, dl2;
  Z_DBInfo *p1, *p2;
  Int16 res;

  dl1 = *((Int32 *) d1);
  dl2 = *((Int32 *) d2);
  p1 = DBI_gDBList[dl1];
  p2 = DBI_gDBList[dl2];

  if(p1->size < p2->size)
  {
    res = -1;
  }
  else if(p1->size > p2->size)
  {
    res = 1;
  }
  else
  {
    res = StrCaselessCompare(p1->name, p2->name);
  }

  if(SORT_gSortReverse)
  {
    res = res * (-1);
  }

  return res;
}

static Int16 SORT_sorter_records(void *d1, void *d2, Int32 rock)
{
  Int32 dl1, dl2;
  Z_DBInfo *p1, *p2;
  Int16 res;

  dl1 = *((Int32 *) d1);
  dl2 = *((Int32 *) d2);
  p1 = DBI_gDBList[dl1];
  p2 = DBI_gDBList[dl2];

  if(p1->records < p2->records)
  {
    res = -1;
  }
  else if(p1->records > p2->records)
  {
    res = 1;
  }
  else
  {
    res = StrCaselessCompare(p1->name, p2->name);
  }

  if(SORT_gSortReverse)
  {
    res = res * (-1);
  }

  return res;
}

static Int16 SORT_sorter_attributes(void *d1, void *d2, Int32 rock)
{
  Int32 dl1, dl2;
  Z_DBInfo *p1, *p2;
  Int16 res;

  dl1 = *((Int32 *) d1);
  dl2 = *((Int32 *) d2);
  p1 = DBI_gDBList[dl1];
  p2 = DBI_gDBList[dl2];

  if(p1->attributes < p2->attributes)
  {
    res = -1;
  }
  else if(p1->attributes > p2->attributes)
  {
    res = 1;
  }
  else
  {
    res = StrCaselessCompare(p1->name, p2->name);
  }

  if(SORT_gSortReverse)
  {
    res = res * (-1);
  }

  return res;
}

static Int16 SORT_sorter_created(void *d1, void *d2, Int32 rock)
{
  Int32 dl1, dl2;
  Z_DBInfo *p1, *p2;
  Int16 res;

  dl1 = *((Int32 *) d1);
  dl2 = *((Int32 *) d2);
  p1 = DBI_gDBList[dl1];
  p2 = DBI_gDBList[dl2];

  if(p1->created < p2->created)
  {
    res = -1;
  }
  else if(p1->created > p2->created)
  {
    res = 1;
  }
  else
  {
    res = StrCaselessCompare(p1->name, p2->name);
  }

  if(SORT_gSortReverse)
  {
    res = res * (-1);
  }

  return res;
}

static Int16 SORT_sorter_modified(void *d1, void *d2, Int32 rock)
{
  Int32 dl1, dl2;
  Z_DBInfo *p1, *p2;
  Int16 res;

  dl1 = *((Int32 *) d1);
  dl2 = *((Int32 *) d2);
  p1 = DBI_gDBList[dl1];
  p2 = DBI_gDBList[dl2];

  if(p1->modified < p2->modified)
  {
    res = -1;
  }
  else if(p1->modified > p2->modified)
  {
    res = 1;
  }
  else
  {
    res = StrCaselessCompare(p1->name, p2->name);
  }

  if(SORT_gSortReverse)
  {
    res = res * (-1);
  }

  return res;
}

static Int16 SORT_sorter_backuped(void *d1, void *d2, Int32 rock)
{
  Int32 dl1, dl2;
  Z_DBInfo *p1, *p2;
  Int16 res;

  dl1 = *((Int32 *) d1);
  dl2 = *((Int32 *) d2);
  p1 = DBI_gDBList[dl1];
  p2 = DBI_gDBList[dl2];

  if(p1->backuped < p2->backuped)
  {
    res = -1;
  }
  else if(p1->backuped > p2->backuped)
  {
    res = 1;
  }
  else
  {
    res = StrCaselessCompare(p1->name, p2->name);
  }

  if(SORT_gSortReverse)
  {
    res = res * (-1);
  }

  return res;
}

static Int16 SORT_sorter_RWRO(void *d1, void *d2, Int32 rock)
{
  Int32 dl1, dl2;
  Z_DBInfo *p1, *p2;
  Int16 res;
  Boolean ro1, ro2;

  dl1 = *((Int32 *) d1);
  dl2 = *((Int32 *) d2);
  p1 = DBI_gDBList[dl1];
  p2 = DBI_gDBList[dl2];

  ro1 = p1->attributes & dmHdrAttrReadOnly;
  ro2 = p2->attributes & dmHdrAttrReadOnly;

  if( ro1 < ro2 )
  {
    res = -1;
  }
  else if( ro1 > ro2 )
  {
    res = 1;
  }
  else
  {
    res = StrCaselessCompare(p1->name, p2->name);
  }

  if(SORT_gSortReverse)
  {
    res = res * (-1);
  }

  return res;
}

static Int16 SORT_sorter_version(void *d1, void *d2, Int32 rock)
{
  Int32 dl1, dl2;
  Z_DBInfo *p1, *p2;
  Int16 res;

  dl1 = *((Int32 *) d1);
  dl2 = *((Int32 *) d2);
  p1 = DBI_gDBList[dl1];
  p2 = DBI_gDBList[dl2];

  if(p1->version < p2->version)
  {
    res = -1;
  }
  else if(p1->version > p2->version)
  {
    res = 1;
  }
  else
  {
    if(p1->modifier < p2->modifier)
    {
      res = -1;
    }
    else if(p1->modifier > p2->modifier)
    {
      res = 1;
    }
    else
    {
      res = StrCaselessCompare(p1->name, p2->name);
    }
  }

  if(SORT_gSortReverse)
  {
    res = res * (-1);
  }

  return res;
}

static Int16 SORT_sorter_card(void *d1, void *d2, Int32 rock)
{
  Int32 dl1, dl2;
  Z_DBInfo *p1, *p2;
  Int16 res;
  Int16 l1, l2;

  dl1 = *((Int32 *) d1);
  dl2 = *((Int32 *) d2);
  p1 = DBI_gDBList[dl1];
  p2 = DBI_gDBList[dl2];

  if(p1->card< p2->card)
  {
    res = -1;
  }
  else if(p1->card> p2->card)
  {
    res = 1;
  }
  else
  {
    l1 = TOOL_location( p1->card, p1->lid );
    l2 = TOOL_location( p2->card, p2->lid );

    if(l1 < l2)
    {
      res = -1;
    }
    else if(l1 > l2)
    {
      res = 1;
    }
    else
    {
      res = StrCaselessCompare(p1->name, p2->name);
    }
  }

  if(SORT_gSortReverse)
  {
    res = res * (-1);
  }

  return res;
}

/* Fill in the DBI_gTabList object with the sorted list of databases. This
   assumes that the DBI_gTabList has already been allocated, and is the
   right size. */
void SORT_build_tabList()
{
  Int32 ix;
  Int32 num;

  char temp1[dmDBNameLength];
  char temp2[dmDBNameLength];

  num = 0;
  for(ix = 0; ix < DBI_gNumDBs; ix++)
  {
    Z_DBInfo *db = DBI_gDBList[ix];

    if(!SORT_gShowReadonly && db->inrom)
    {
      continue;
    }

    if(SORT_gFilter.active)
    {
      Boolean keep;

      switch (SORT_gFilter.type)
      {
        case Z_dColumnTypeCreator:
          keep = (db->creator == SORT_gFilter.u.ostype);
          if(!SORT_gFilter.test)
          {
            keep = !keep;
          }
          break;

        case Z_dColumnTypeType:
          keep = (db->type == SORT_gFilter.u.ostype);
          if(!SORT_gFilter.test)
          {
            keep = !keep;
          }
          break;

        case Z_dColumnTypeSize:
          if(SORT_gFilter.test)
          {
            keep = (db->size <= SORT_gFilter.u.num);
          }
          else
          {
            keep = (db->size >= SORT_gFilter.u.num);
          }
          break;

        case Z_dColumnTypeRecords:
          if(SORT_gFilter.test)
          {
            keep = (db->records <= SORT_gFilter.u.num);
          }
          else
          {
            keep = (db->records >= SORT_gFilter.u.num);
          }
          break;

        case Z_dColumnTypeAttributes:
          keep = (db->attributes <= SORT_gFilter.u.num);  //### erm
          break;

        case Z_dColumnTypeName:
          if(SORT_gFilter.test == 0)
          {
            Int16 len = StrLen(SORT_gFilter.u.str);
            keep = !StrNCaselessCompare(db->name, SORT_gFilter.u.str, len);
          }
          else if(SORT_gFilter.test == 2)
          {
            StrToLower(temp1, db->name);
            StrToLower(temp2, SORT_gFilter.u.str);
            keep = (StrStr(temp1, temp2) != NULL);
          }
          else
          {
            Int16 len = StrLen(SORT_gFilter.u.str);
            Int16 len2 = StrLen(db->name);
            if(len2 < len)
            {
              keep = FALSE;
            }
            else
            {
              keep = !StrCaselessCompare(db->name + (len2 - len), SORT_gFilter.u.str);
            }
          }
          break;

        default:
          keep = TRUE;
          break;
      }

      if(!keep)
      {
        continue;
      }
    }

    DBI_gTabList[num] = ix;
    num++;
  }

  if(num >= 2)
  {
    CmpFuncPtr sortfn;

    switch (SORT_gSortType)
    {
      case Z_dColumnTypeType:
        sortfn = SORT_sorter_type;
        break;

      case Z_dColumnTypeCreator:
        sortfn = SORT_sorter_creator;
        break;

      case Z_dColumnTypeSize:
        sortfn = SORT_sorter_size;
        break;

      case Z_dColumnTypeRecords:
        sortfn = SORT_sorter_records;
        break;

      case Z_dColumnTypeAttributes:
        sortfn = SORT_sorter_attributes;
        break;

      case Z_dColumnTypeCreated:
      case Z_dColumnTypeCreatedTime:
        sortfn = SORT_sorter_created;
        break;

      case Z_dColumnTypeModified:
      case Z_dColumnTypeModifiedTime:
        sortfn = SORT_sorter_modified;
        break;

      case Z_dColumnTypeBackuped:
      case Z_dColumnTypeBackupedTime:
        sortfn = SORT_sorter_backuped;
        break;

      case Z_dColumnTypeRWRO:
        sortfn = SORT_sorter_RWRO;
        break;

      case Z_dColumnTypeVersion:
        sortfn = SORT_sorter_version;
        break;

      case Z_dColumnTypeCard:
        sortfn = SORT_sorter_card;
        break;

      default:
        sortfn = SORT_sorter_name;
        break;
    }

    SysQSort(DBI_gTabList, num, sizeof(long), sortfn, NULL);
  }

  DBI_gNumTabs = num;
}

/* Create and fill in the "result" with the list of databases with "creator". This
   assumes that the "result" has already been allocated, and is the
   right size. */
Int32 SORT_build_list_for_creator(UInt32 creator, UInt32 * result)
{
  Int32 ix;
  Int32 num;

  num = 0;
  for(ix = 0; ix < DBI_gNumDBs; ix++)
  {
    Z_DBInfo *db = DBI_gDBList[ix];

    if(!SORT_gShowReadonly && db->inrom)
    {
      continue;
    }

    if(db->creator != creator)
    {
      continue;
    }

    result[num] = ix;
    num++;
  }

  return num;
}
