/*
   Tom'Catalog
   Copyright 2002, Thomas Aufischer

   based on:

   Z'Catalog: a PalmOS database utility
   Copyright 1998, Andrew Plotkin <erkyrath@eblong.com>
   http://www.eblong.com/zarf/zcatalog.html

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <PalmOS.h>
#include "zcatalog.h"
#include "resource.h"

Int32 DBI_gNumDBs = 0;
// we keep now pointers to Z_DBInfo
Z_DBInfo **DBI_gDBList = NULL;
// this is the handle to the array of pointers
static MemHandle dbhan = NULL;
// we keep now additionally a array of handles (for each Z_DBInfo)
static MemHandle hanhan = NULL;
MemHandle *hanList = NULL;

Int32 DBI_gNumTabs = 0;
Int32 *DBI_gTabList = NULL;
static MemHandle tabhan = NULL;

#define dTotalWidth (120)

static void DBI_real_DBI_load_dbinfo(void);

/* Get the DBI_gTabList object all allocated and loaded up with juicy info.
   This is a wrapper that displays a "working..." dialog. */
void DBI_load_dbinfo()
{
  Err err;
  RectangleType winbox = { {10, 40}, {140, 34} };
  RectangleType barbox = { {10, 19}, {dTotalWidth, 8} };
  WinHandle oldwin = NULL;
  WinHandle progwin = NULL;

  progwin = WinCreateWindow(&winbox, dialogFrame, TRUE, FALSE, &err);
  if(progwin)
  {
    FontID oldfont;
    oldwin = WinGetDrawWindow();
    WinSetDrawWindow(progwin);
    WinEraseWindow();
    WinDrawWindowFrame();
    oldfont = FntGetFont();
    FntSetFont(stdFont);
    WinDrawChars("Scanning databases....", 22, 5, 3);
    FntSetFont(oldfont);
    WinDrawRectangleFrame(simpleFrame, &barbox);
  }

  DBI_real_DBI_load_dbinfo();

  if(progwin)
  {
    WinDeleteWindow(progwin, TRUE);
    WinSetDrawWindow(oldwin);
  }
}

/* Get the DBI_gTabList object all allocated and loaded up with juicy info. */
static void DBI_real_DBI_load_dbinfo()
{
  RectangleType barbox = { {10, 19}, {0, 8} };
  Err err;
  Int16 numcards;
  Int16 card;
  Int32 dbi;
  char errorText[100];

  /* Free up all the old data, if there is any. */
  DBI_unload_dbinfo();

  /* First, we count. Then, we allocate. */
  numcards = MemNumCards();
  for(card = 0; card < numcards; card++)
  {
    err = MemCardInfo(card, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
    if(!err)
    {
      UInt16 num;
      num = DmNumDatabases(card);
      DBI_gNumDBs += num;
    }
    else
    {
      StrPrintF(errorText, "MemCardInfo failed for Card %d, error %d", card, err);
      UTIL_printError( errorText );
    }
  }

  /* Well, if there aren't any, I guess we're done. */
  if(DBI_gNumDBs == 0)
  {
    UTIL_printError( "Sorry, no database found by use of Memory Manager" );
    return;
  }

  /* Now allocate the space we need. */
  dbhan = MemHandleNew(DBI_gNumDBs * sizeof(Z_DBInfo *));
  if(!dbhan)
  {
    StrPrintF(errorText, "Error at allocating %d bytes -A- probably you have too less free memory",
                         DBI_gNumDBs * sizeof(Z_DBInfo *) );
    UTIL_printError( errorText );

    DBI_gNumDBs = 0;
    return;
  }

  tabhan = MemHandleNew(DBI_gNumDBs * sizeof(long));
  if(!tabhan)
  {
    StrPrintF(errorText, "Error at allocating %d bytes -B- probably you have too less free memory",
                         DBI_gNumDBs * sizeof(long) );
    UTIL_printError( errorText );

    MemHandleFree(dbhan);

    DBI_gNumDBs = 0;
    return;
  }

  hanhan = MemHandleNew(DBI_gNumDBs * sizeof(MemHandle));
  if(!hanhan)
  {
    StrPrintF(errorText, "Error at allocating %d bytes -C- probably you have too less free memory",
                         DBI_gNumDBs * sizeof(MemHandle) );
    UTIL_printError( errorText );

    MemHandleFree(dbhan);
    MemHandleFree(tabhan);

    DBI_gNumDBs = 0;
    return;
  }


  DBI_gDBList = MemHandleLock(dbhan);
  DBI_gTabList = MemHandleLock(tabhan);
  hanList = MemHandleLock(hanhan);
  if(!DBI_gDBList || !DBI_gTabList || !hanList )
  {
    UTIL_printError( "Error at MemHandleLock -A-");

    MemHandleFree(dbhan);
    MemHandleFree(tabhan);
    MemHandleFree(hanhan);

    DBI_gNumDBs = 0;
    return;
  }

  dbi = 0;

  for(card = 0; card < numcards; card++)
  {
    err = MemCardInfo(card, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
    if(!err)
    {
      UInt16 ix, num;

      num = DmNumDatabases(card);

      for(ix = 0; ix < num; ix++)
      {
        LocalID lid;
        Z_DBInfo *db = NULL;

        if( dbi >= DBI_gNumDBs )
        {
          StrPrintF(errorText, "Internal Error - found initially %d database but now I find more",
                               DBI_gNumDBs );
          UTIL_printError( errorText );
          return;
        }

        lid = DmGetDatabase(card, ix);
        if(lid)
        {
          // Now allocate the space we need for Z_DBInfo
          MemHandle han = MemHandleNew(sizeof(Z_DBInfo));
          if(!han)
          {
            StrPrintF(errorText, "Error at allocating %d bytes -D- probably you have too less free memory",
                                 sizeof(Z_DBInfo) );
            UTIL_printError( errorText );

            DBI_gNumDBs = dbi;
            return;
          }

          db = MemHandleLock(han);
          if(!db)
          {
            UTIL_printError( "Error at MemHandleLock -B-");
            MemHandleFree(han);
            DBI_gNumDBs = dbi;
            return;
          }

          // remember handle so that we can free it lateron
          hanList[dbi] = han;
          // remember pointer
          DBI_gDBList[dbi] = db;


          db->card = card;
          db->lid = lid;

          err = DmDatabaseInfo( card, lid,
                                db->name, &(db->attributes), &(db->version),
                                &(db->created), &(db->modified), &db->backuped,
                                &(db->modifier), NULL, NULL,
                                &(db->type), &(db->creator) );
          if( err )
          {
            StrPrintF(errorText, "DmDatabaseInfo failed for Card %d, localeId %d, error %d", card, lid, err);
            UTIL_printError( errorText );
          }

          err = DmDatabaseSize(card, lid, &(db->records), &(db->size), NULL);
          if( err )
          {
            StrPrintF(errorText, "DmDatabaseSize failed for Card %d, localeId %d, error %d", card, lid, err);
            UTIL_printError( errorText );
          }

          db->inrom = ( TOOL_location(card, lid) == TOOL_eLocation_ROM );
        }
        else
        {
          StrPrintF(errorText, "DmGetDatabasefailed for Card %d, index %d, error %d", card, ix, err);
          UTIL_printError( errorText );
        }

        dbi++;

        if((dbi % 10) == 9)
        {
          barbox.extent.x = ((long) dTotalWidth * dbi) / (long) DBI_gNumDBs;
          WinDrawRectangle(&barbox, 0);
        }
      }
    }
    else
    {
      // was already treated above
    }
  }

  /* Fill in the data. */
  SORT_build_tabList();
}

/* Deallocate the DBI_gTabList object. */
void DBI_unload_dbinfo()
{
  int i;

  if(tabhan)
  {
    MemHandleFree(tabhan);
    tabhan = NULL;
  }
  DBI_gTabList = NULL;
  DBI_gNumTabs = 0;

  if(hanhan)
  {
    /* Free the old data. */
    for( i = 0; i < DBI_gNumDBs; i++ )
    {
      MemHandleFree(hanList[i]);
    }

    MemHandleFree(hanhan);
    hanhan = NULL;
  }

  if(dbhan)
  {
    /* Free the old data. */
    MemHandleFree(dbhan);
    dbhan = NULL;
  }
  DBI_gDBList = NULL;
  DBI_gNumDBs = 0;
}

Int32 DBI_find_dbi_by_name(char *name)
{
  Int32 ix;
  for(ix = 0; ix < DBI_gNumDBs; ix++)
  {
    if(!StrCompare(name, DBI_gDBList[ix]->name))
    {
      return ix;
    }
  }
  return -1;
}
