/*
   Tom'Catalog
   Copyright 2002, Thomas Aufischer

   based on:

   Z'Catalog: a PalmOS database utility
   Copyright 1998, Andrew Plotkin <erkyrath@eblong.com>
   http://www.eblong.com/zarf/zcatalog.html

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <PalmOS.h>
#include <ExgMgr.h>
#include <FloatMgr.h>
#include "zcatalog.h"
#include "resource.h"

#define TOOL_dFlashHeapId 4

Int16 TOOL_location(UInt16 card, LocalID lid)
{
  if (MemLocalIDKind(lid) == memIDHandle)
  {
    return TOOL_eLocation_RAM;
  }
  else if (MemPtrHeapID(MemLocalIDToPtr(lid, card)) == TOOL_dFlashHeapId )
  {
    return TOOL_eLocation_FLASH;
  }
  else
  {
    return TOOL_eLocation_ROM;
  }
}

const char* TOOL_locationString(UInt16 card, LocalID lid)
{
  return Z_cLocationString[TOOL_location(card, lid)];
}

/* Test if a DB is in ROM. This should really work now ?!?
Boolean TOOL_test_in_rom(UInt16 card, LocalID lid)
{
  void *phan;
  UInt16 heapid;

  phan = MemLocalIDToGlobal(lid, card);
  if(MemLocalIDKind(lid) == memIDHandle)
  {
    heapid = MemHandleHeapID(phan);
  }
  else
  {
    heapid = MemPtrHeapID(phan);
  }

  if(MemHeapFlags(heapid) & 0x0001) // memHeapFlagReadOnly
  {
    return TRUE;
  }
  else
  {
    return FALSE;
  }
}
*/

char* TOOL_getRWString( Int32 dbi, char* buffer )
{
  if(DBI_gDBList[dbi]->attributes & dmHdrAttrReadOnly)
  {
    return StrCopy( buffer , "RO" );
  }

  return  StrCopy( buffer , "RW" );
}

/* Try to change a database's attributes, and show errors if there are
   any. Returns TRUE if it worked, FALSE if not (or cancelled). */
Boolean TOOL_change_prop_db(Int16 changes, Z_Changedata * newdat,
                            UInt16 card, LocalID lid)
{
  Err err;
  Int16 val, ix, jx, count;
  char buf[64];
  char *newname;
  UInt32 *newtype, *newcreator;
  UInt32 *newcreatedate, *newmoddate, *newbackupdate;
  UInt16 *newattr;

  if(!changes)
  {
    return TRUE;
  }

  if(!MAIN_gBoneheadmode)
  {
    FrmAlert(alertid_NonBoneheadMode);
    return FALSE;
  }

  if( TOOL_location(card, lid) == TOOL_eLocation_ROM )
  {
    FrmAlert(alertid_CannotApplyDBChangesROM);
    return FALSE;
  }

  for(val = changes, count = 0; val; val >>= 1)
  {
    if(val & 1)
      count++;
  }

  /* Generate a string that tells what changes are being made. */
  buf[0] = '\0';
  for(val = changes, ix = 0, jx = 0; val; val >>= 1, jx++)
  {
    if(val & 1)
    {
      if(ix != 0)
      {
        if(ix == count - 1)
        {
          StrCat(buf, " and ");
        }
        else
        {
          StrCat(buf, ", ");
        }
      }
      switch (jx)
      {
        case 0:
          StrCat(buf, "name");
          break;
        case 1:
          StrCat(buf, "attributes");
          break;
        case 2:
          StrCat(buf, "type");
          break;
        case 3:
          StrCat(buf, "creator");
          break;
        case 4:
          StrCat(buf, "timestamps");
          break;
      }
      ix++;
    }
  }

  val = FrmCustomAlert(alertid_ApplyDBChanges, buf, " ", " ");
  if(val == 1)
  {
    return FALSE;
  }

  /* The user has okayed the changes. */

  newname = NULL;
  newattr = NULL;
  newtype = NULL;
  newcreator = NULL;
  newcreatedate = NULL;
  newmoddate = NULL;
  newbackupdate = NULL;

  if(changes & Z_dChangedName)
  {
    newname = newdat->name;
  }
  if(changes & Z_dChangedAttr)
  {
    newattr = &newdat->attributes;
  }
  if(changes & Z_dChangedType)
  {
    newtype = &newdat->type;
  }
  if(changes & Z_dChangedCreator)
  {
    newcreator = &newdat->creator;
  }
  if(changes & Z_dChangedTimes)
  {
    newcreatedate = &newdat->create_date;
    newmoddate = &newdat->mod_date;
    newbackupdate = &newdat->backup_date;
  }

  err = DmSetDatabaseInfo(card, lid,
                          newname, newattr,
                          NULL,
                          newcreatedate, newmoddate, newbackupdate,
                          NULL, NULL, NULL, newtype, newcreator);
  if(err)
  {
    FrmAlert(alertid_CannotApplyDBChanges);
    return FALSE;
  }

  MAIN_mainform_dirty_dbist(1);

  return TRUE;
}

/* Try to delete a database, and show errors if there are any. Returns
   TRUE if it worked, FALSE if not (or if cancelled). */
Boolean TOOL_delete_whole_db(UInt16 card, LocalID lid,
                             UInt32 type, Int16 showAlert)
{
  Int16 ix, res;
  UInt16 attr;
  Err err;

  if(!MAIN_gBoneheadmode)
  {
    FrmAlert(alertid_NonBoneheadMode);
    return FALSE;
  }

  if( TOOL_location(card, lid) == TOOL_eLocation_ROM )
  {
    FrmAlert(alertid_CannotDeleteDBROM);
    return FALSE;
  }

  // show alert?
  if(showAlert > 0)
  {
    // delete one DB/appl
    if(showAlert == 1)
    {
      if(type == 'appl')
      {
        ix = alertid_ReallyDeleteDBAppl;
      }
      else
      {
        ix = alertid_ReallyDeleteDB;
      }
    }
    // delete more DB/appl
    else
    {
      ix = alertid_ReallyDeleteMoreDBAppl;
    }

    res = FrmAlert(ix);
    if(res == 1)
    {
      return FALSE;                 /* cancel */
    }
  }

  /* The user has okayed the deletion. */

  err = DmDatabaseInfo(card, lid, NULL, &attr,
                       NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);

  if(!err && (attr & dmHdrAttrReadOnly))
  {
    attr &= (~dmHdrAttrReadOnly);
    err = DmSetDatabaseInfo(card, lid, NULL, &attr,
                            NULL, NULL, NULL, NULL,
                            NULL, NULL, NULL, NULL, NULL);
    if(err)
    {
      FrmCustomAlert(alertid_CannotDeleteDB,
                     "because the read-only flag could not be turned off",
                     " ", " ");
      return FALSE;                 /* error */
    }
  }

  err = DmDeleteDatabase(card, lid);
  if(err)
  {
    char *str;

    switch (err)
    {
      case dmErrCantFind:
        str = "because it doesn't seem to exist";
        break;
      case dmErrCantOpen:
        str = "because it could not be opened";
        break;
      case memErrChunkLocked:
        str = "because a chunk was locked";
        break;
      case dmErrDatabaseOpen:
        str = "because it is currently open";
        break;
      case dmErrROMBased:
        str = "because it is in ROM";
        break;
      case memErrNotEnoughSpace:
        str = "because you are low on memory (doesn't make sense to me either)";
        break;
      default:
        str = "for some unknown reason";
        break;
    }

    FrmCustomAlert(alertid_CannotDeleteDB, str, " ", " ");

    return FALSE;                   /* error */
  }

  MAIN_mainform_dirty_dbist(2);

  return TRUE;
}

/* Utility callbacks for receiving a beamed DB. */

static Err TOOL_readthunk(void *datap, UInt32 * sizep, void *rock)
{
  Err err = 0;
  ExgSocketType *sock = rock;
  UInt32 count;

  count = *sizep;
  count = ExgReceive(sock, datap, count, &err);
  *sizep = count;

  return err;
}

static Boolean TOOL_deletethunk(const char *name, UInt16 version, Int16 card,
                                LocalID lid, void *rock)
{
  Err err;
  char newname[dmDBNameLength], suffixbuf[dmDBNameLength];
  Int16 len, pos, len2;
  Boolean hadold = FALSE;
  Boolean hadnum = FALSE;
  Int16 vernum, verdigit;
  UInt16 attr;

  err = DmDatabaseInfo(card, lid, NULL, &attr,
                       NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
  if(err)
    return FALSE;

  if( TOOL_location(card, lid) == TOOL_eLocation_ROM )
  {
    return FALSE;
  }

  /* Generate an alternate name. If "zob" already exists, for example,
     rename it to "zob old", so that a new "zob" can be received. */

  len = StrLen(name);
  StrCopy(newname, name);

  pos = len;
  vernum = 0;
  verdigit = 1;
  while(pos > 0 && newname[pos - 1] >= '0' && newname[pos - 1] <= '9')
  {
    vernum += verdigit * (newname[pos - 1] - '0');
    verdigit *= 10;
    hadnum = TRUE;
    pos--;
  }

  while(pos > 0 && newname[pos - 1] == ' ')
    pos--;

  if(pos > 3
     && newname[pos - 1] == 'd'
     && newname[pos - 2] == 'l' && newname[pos - 3] == 'o')
  {
    pos -= 3;
    hadold = TRUE;
    while(pos > 0 && newname[pos - 1] == ' ')
      pos--;
  }

  if(hadold)
  {
    len = pos;
  }

  StrCopy(suffixbuf, " old");
  if(hadold)
  {
    if(!hadnum)
    {
      vernum = 2;
    }
    else
    {
      vernum++;
    }
    suffixbuf[4] = ' ';
    StrIToA(suffixbuf + 5, vernum);
  }
  len2 = StrLen(suffixbuf);

  if(len > 31 - len2)
  {
    len = 31 - len2;
  }
  StrCopy(newname + len, suffixbuf);

  /* Rename. */

  err = DmSetDatabaseInfo(card, lid, newname, NULL,
                          NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
  if(err)
  {
    return FALSE;
  }

  return TRUE;
}

Err TOOL_beam_receive_db(ExgSocketPtr sock, Boolean globavail)
{
  Err err = 0;
  LocalID lid;
  UInt16 card = 0;
  Boolean wantreset = FALSE;

  /* necessary for beaming to PalmOS 4.0 and later, thanks to Patrice Bernard */
  err= ExgAccept( sock );

  err = ExgDBRead(TOOL_readthunk, (ExgDBDeleteProcPtr) TOOL_deletethunk, sock, &lid,
                  card, &wantreset, TRUE);
  err = ExgDisconnect(sock, err);

  sock->goToCreator = 0;

  if(err)
    return err;

  if(wantreset)
  {
    Int16 res = FrmAlert(alertid_WantReset);
    if(res == 1)
      SysReset();
  }

  /*
     if (globavail) {
     MAIN_mainform_notice_new_db(card, lid);
     }
   */

  return err;
}

/* Utility callback for beaming out a DB. */

static Err TOOL_writethunk(const void *datap, UInt32 * sizep, void *rock)
{
  Err err = 0;
  ExgSocketType *sock = rock;
  UInt32 count;

  count = *sizep;
  count = ExgSend(sock, datap, count, &err);
  *sizep = count;

  return err;
}

Err TOOL_beam_send_db(Z_DBInfo * db)
{
  ExgSocketType sock;
  Err err;

  MemSet(&sock, sizeof(ExgSocketType), 0);
  sock.description = db->name;
  sock.type = Z_dSelfMimetype;
  sock.name = db->name;
  sock.count = 1;
  sock.length = db->size;
  sock.target = Z_dSelfCreator;

  err = ExgPut(&sock);
  if(err)
  {
    char *str;
    switch (err)
    {
      case exgErrBadLibrary:
        str = "because the exchange library couldn't be found";
        break;
      case exgErrStackInit:
        str = "because the IR protocol couldn't be initialized";
        break;
      case exgMemError:
        str = "because there wasn't sufficient memory";
        break;
      default:
        str = "for some unknown reason";
        break;
    }
    FrmCustomAlert(alertid_CannotBeamDB, str, " ", " ");

    return err;
  }

  err = ExgDBWrite(TOOL_writethunk, &sock, db->name, db->lid, db->card);

  err = ExgDisconnect(&sock, err);
  if(err && err != exgErrUserCancel)
  {
    FrmCustomAlert(alertid_CannotBeamDB, "for some unknown reason", " ", " ");
  }

  return err;
}

char *TOOL_deleteChar(char *input, Int16 * len, char toDelete)
{
  int i = 0;

  for(;;)
  {
    if(input[i] == '\0')
    {
      *len = StrLen(input);
      return input;
    }

    if(input[i] == toDelete)
    {
      StrCopy(input + i, input + i + 1);
    }
    else
    {
      i++;
    }
  }
}

char *TOOL_replaceChar(char *input, Int16 * len, char from, char to)
{
  int i = 0;

  for(;;)
  {
    if(input[i] == '\0')
    {
      *len = StrLen(input);
      return input;
    }

    if(input[i] == from)
    {
      input[i] = to;
    }

    i++;
  }
}

void TOOL_fill_out_temp(Z_DateTime * tem)
{
  if(tem->base == 0)
  {
    tem->active = FALSE;
    MemSet(&tem->fields, sizeof(DateTimeType), 0);
  }
  else
  {
    tem->active = TRUE;
    TimSecondsToDateTime(tem->base, &tem->fields);
  }
}

UInt32 TOOL_temp_to_base(Z_DateTime * tem)
{
  if(!tem->active)
  {
    return 0;
  }

  return TimDateTimeToSeconds(&tem->fields);
}

void TOOL_dateTimeString(Z_DateTime * tem)
{
  DateFormatType dateFormat = dfMDYWithSlashes;
  TimeFormatType timeFormat = tfColonAMPM;

  if(!tem->active)
  {
    StrCopy(tem->date_buf, "Never");
    StrCopy(tem->time_buf, "Never");
    return;
  }

  if(!MAIN_gUSDateTime)
  {
    dateFormat = dfYMDWithDashes;
    timeFormat = tfColon24h;
  }

  DateToAscii(tem->fields.month, tem->fields.day, tem->fields.year,
              dateFormat, tem->date_buf);
  TimeToAscii(tem->fields.hour, tem->fields.minute, timeFormat, tem->time_buf);
}

char *TOOL_dateStringShort(UInt32 dateTimeInt, char *helpBuffer, Int16 * len)
{
  DateTimeType dateTime;
  char year[5];

  if(dateTimeInt == 0)
  {
    StrCopy(helpBuffer, "Never");
    *len = StrLen(helpBuffer);
    return helpBuffer;
  }

  TimSecondsToDateTime(dateTimeInt, &dateTime);
  StrCopy(year, StrIToA(year, dateTime.year) + 2);
  if(MAIN_gUSDateTime)
  {
    StrPrintF(helpBuffer, "%2d%2d%2s", dateTime.month, dateTime.day, year);
  }
  else
  {
    StrPrintF(helpBuffer, "%2s%2d%2d", year, dateTime.month, dateTime.day);
  }
  TOOL_replaceChar(helpBuffer, len, ' ', '0');

  return helpBuffer;
}

char *TOOL_timeStringShort(UInt32 dateTimeInt, char *helpBuffer, Int16 * len)
{
  DateTimeType dateTime;
  char help[10];

  if(dateTimeInt == 0)
  {
    StrCopy(helpBuffer, "None");
    *len = StrLen(helpBuffer);
    return helpBuffer;
  }

  TimSecondsToDateTime(dateTimeInt, &dateTime);

  if(MAIN_gUSDateTime)
  {
    TimeToAscii(dateTime.hour, dateTime.minute, tfColonAMPM, helpBuffer);
    TOOL_deleteChar(helpBuffer, len, ' ');
  }
  else
  {
    TimeToAscii(dateTime.hour, dateTime.minute, tfColon24h, helpBuffer);
  }

  if(helpBuffer[1] == ':')
  {
    StrCopy(help, helpBuffer);
    StrPrintF(helpBuffer, "0%s", help);
  }

  *len = StrLen(helpBuffer);

  return helpBuffer;
}

void TOOLS_cvtSecondsToTime( UInt32 seconds, DateTimeType *deltaTime )
{
  UInt32 help;

  MemSet( deltaTime, 0, sizeof(DateTimeType) );

  help         = seconds % (60*60);
  deltaTime->hour   = seconds / (60*60);

  seconds      = help;
  help         = seconds % 60;
  deltaTime->minute = seconds / 60;

  deltaTime->second = help;
}

void TOOL_getCardInfo( char *text )
{
  char helpBuffer[100];
  Int16 numcards;
  Err err;
  Int16 card;
  Int32 dbi;
  char errorText[100];
  char cardName[32];
  char manufName[32];
  UInt16 version;
  Z_DateTime crDateProp;
  UInt32 romSize;
  UInt32 ramSize;
  UInt32 freeBytes;
  UInt16 num;

  StrCopy( text, "\n" );

  numcards = MemNumCards();
  StrPrintF(helpBuffer, "%d card(s) found:\n", numcards);
  StrCat( text, helpBuffer );

  for(card = 0; card < numcards; card++)
  {
    StrPrintF(helpBuffer, "\nCard %d : ", card);
    StrCat( text, helpBuffer );

    err = MemCardInfo(card, cardName, manufName, &version, &(crDateProp.base), &romSize, &ramSize, &freeBytes );
    if(!err)
    {
      StrPrintF(helpBuffer, "%s\n", cardName );
      StrCat( text, helpBuffer );

      StrPrintF(helpBuffer, "Manufacturer: %s\n", manufName );
      StrCat( text, helpBuffer );

      StrPrintF(helpBuffer, "Version: %d\n", version );
      StrCat( text, helpBuffer );

      TOOL_fill_out_temp(&crDateProp);
      TOOL_dateTimeString(&crDateProp);
      StrPrintF(helpBuffer, "Created: %s %s\n", crDateProp.date_buf, crDateProp.time_buf );
      StrCat( text, helpBuffer );

      StrPrintF(helpBuffer, "ROM size: %ld\n", romSize);
      StrCat( text, helpBuffer );

      StrPrintF(helpBuffer, "RAM size: %ld\n", ramSize);
      StrCat( text, helpBuffer );

      StrPrintF(helpBuffer, "Free byte: %ld\n", freeBytes);
      StrCat( text, helpBuffer );
    }
    else
    {
      StrCat( text, "MemCardInfo failed" );
    }

    num = DmNumDatabases(card);
    StrPrintF(helpBuffer, "Databases: %d\n", num);
    StrCat( text, helpBuffer );
  }
}

void TOOL_getSystemInfo( char *text )
{
  char helpBuffer[100];
  Err err;
  char errorText[100];
  Char *osVersion;
  UInt8 *dataP;
  UInt16 size;
  UInt32 seconds;
  DateTimeType deltaTime;
  UInt32 romVersion;
  UInt32 id, chip;
  UInt16 revision;

  StrCopy( text, "\n" );
  StrCat( text, "PalmOS version : " );

  if( ! MAIN_gOS3_avail )
  {
    StrCat( text, "< 3.0\n" );
  }
  else
  {
    osVersion = SysGetOSVersionString();
    StrCat( text, osVersion );
    StrCat( text, "\n" );
    MemPtrFree( osVersion );

    StrCat( text, "ROM serial : " );
    err = SysGetROMToken (0, sysROMTokenSnum, &dataP, &size);
    if( ( !err ) && ( dataP != NULL ) && ( size > 0 ) && ( dataP[0] != 0xFF ) )
    {
      StrNCopy (helpBuffer, (Char *) dataP, size);
      helpBuffer[size] = '\n';
      helpBuffer[size+1] = '\0';
      StrCat( text, helpBuffer );
    }
    else
    {
      StrCat( text, "not available\n" );
    }
  }

  StrCat( text, "System up : " );
  seconds = TimGetTicks() / SysTicksPerSecond();
  TOOLS_cvtSecondsToTime( seconds, &deltaTime );
  StrPrintF(helpBuffer, "%dh %dm %ds\n",
            deltaTime.hour, deltaTime.minute, deltaTime.second);
  StrCat( text, helpBuffer );


  err = FtrGet(sysFtrCreator, sysFtrNumROMVersion, &romVersion);
  if( ! err )
  {
    if( romVersion >= sysMakeROMVersion(3,1,0,sysROMStageRelease,0) )
    {
      err = FtrGet(sysFtrCreator, sysFtrNumProcessorID, &id);
      if (!err)
      {
        StrCat( text, "Processor : " );

        chip = id & sysFtrNumProcessorMask;
        revision = id & 0x0ffff;

        if (chip==sysFtrNumProcessor328)
        {
          StrCat( text, "trad. DragonBall (68328)\n" );
	    }
        else if (chip==sysFtrNumProcessorEZ)
        {
          StrCat( text, "DragonBall EZ (68EZ328)\n" );
	    }
        else if (chip==sysFtrNumProcessorVZ)
        {
          StrCat( text, "DragonBall VZ (68VZ328)\n" );
	    }
        else
        {
          StrCat( text, "?\n" );
	    }
      }
    }
  }
}